#include <infra/netmon/topology/topology.h>
#include <infra/netmon/topology/metrics.h>
#include <infra/netmon/topology/settings.h>

#include <util/generic/algorithm.h>
#include <util/generic/maybe.h>
#include <util/stream/file.h>
#include <util/stream/zlib.h>
#include <util/string/join.h>
#include <util/string/subst.h>

namespace NNetmon {
    namespace {
        const std::size_t CHUNK_SIZE = 8192;
        const TStringBuf UNKNOWN = "unknown";

        class TObjectFactory {
        public:
            enum class EParsedNetworkType {
                BACKBONE,
                FASTBONE
            };

            TObjectFactory(TTopologyIndexes& indexes,
                           TExistedNames& existedNames)
                : Indexes(indexes)
                , ExistedNames(existedNames)
            {
            }

            void AssignString(const TString*& dest, msgpack::object_kv& ptr) {
                const TString value(ptr.val.as<TStringBuf>());
                if (!value.empty() && value != UNKNOWN) {
                    dest = &ExistedNames.Intern(value);
                }
            }

            void AssignString(TMaybe<TString>& dest, msgpack::object_kv& ptr) {
                const TString value(ptr.val.as<TStringBuf>());
                if (!value.empty() && value != UNKNOWN) {
                    dest = ExistedNames.Intern(value);
                }
            }

            void AssignIpAddress(TMaybe<TIpAddress>& dest, int family, msgpack::object_kv& ptr) {
                const TString value(ptr.val.as<TStringBuf>());
                if (!value.empty() && value != UNKNOWN) {
                    dest.ConstructInPlace(family, value);
                }
            }

            void AssignStringList(TStringSet& dest, msgpack::object_kv& ptr) {
                for (auto& x : TListIterator(ptr.val.via.array)) {
                    const TString value(x.as<TStringBuf>());
                    dest.emplace(ExistedNames.Intern(value));
                }
            }

            bool ProcessItem(msgpack::object_kv& ptr) {
                TStringBuf key = ptr.key.as<TStringBuf>();
                if (key == TStringBuf("fqdn")) {
                    AssignString(HostName, ptr);
                } else if (key == TStringBuf("dc")) {
                    AssignString(DatacenterName, ptr);
                } else if (key == TStringBuf("queue")) {
                    if (!TTopologySettings::Get()->GetUsePodAsQueue())
                        AssignString(QueueName, ptr);
                } else if (key == TStringBuf("pod")) {
                    if (TTopologySettings::Get()->GetUsePodAsQueue())
                        AssignString(QueueName, ptr);
                    AssignString(PodName, ptr);
                } else if (key == TStringBuf("switch")) {
                    AssignString(SwitchName, ptr);
                } else if (key == TStringBuf("network_type")) {
                    const auto network(ptr.val.as<TStringBuf>());
                    if (network == TStringBuf("backbone")) {
                        NetworkType = EParsedNetworkType::BACKBONE;
                    } else if (network == TStringBuf("fastbone")) {
                        NetworkType = EParsedNetworkType::FASTBONE;
                    }
                } else if (key == TStringBuf("invnum")) {
                    const auto value(ptr.val.as<TStringBuf>());
                    if (!value.empty() && value != UNKNOWN) {
                        InventoryNumber = FromString(value);
                    }
                } else if (key == TStringBuf("vlan")) {
                    Vlan = ptr.val.as<int>();
                } else if (key == TStringBuf("vrf")) {
                    AssignString(Vrf, ptr);
                } else if (key == TStringBuf("ipv6addr")) {
                    AssignIpAddress(Ipv6, AF_INET6, ptr);
                } else if (key == TStringBuf("ipv4addr")) {
                    AssignIpAddress(Ipv4, AF_INET, ptr);
                } else if (key == TStringBuf("owners") && ptr.val.type == msgpack::type::ARRAY) {
                    AssignStringList(Owners, ptr);
                } else if (key == TStringBuf("walle_project")) {
                    AssignString(WalleProject, ptr);
                } else if (key == TStringBuf("walle_tags") && ptr.val.type == msgpack::type::ARRAY) {
                    AssignStringList(WalleTags, ptr);
                } else {
                    return false;
                }
                return true;
            }

            const THost* CreateHost() {
                const TSwitch* switch_(CreateSwitch());
                if (!switch_ || !HostName) {
                    return nullptr;
                }

                const THost& host(switch_->CreateHost(
                    *HostName, InventoryNumber, Owners, WalleProject, WalleTags,
                    ExistedNames
                ));

                Indexes.HostIdSet.insert(&host);
                Indexes.HostNameSet.insert(&host);

                return &host;
            }

            const THostInterface* CreateHostInterface(const THost& host) {
                ENetworkType ifaceNetworkType = NIL_NETWORK;
                if (NetworkType == EParsedNetworkType::FASTBONE && Ipv6) {
                    ifaceNetworkType = FASTBONE6;
                } else if (NetworkType == EParsedNetworkType::BACKBONE && Ipv6) {
                    ifaceNetworkType = BACKBONE6;
                } else if (NetworkType == EParsedNetworkType::BACKBONE && Ipv4) {
                    ifaceNetworkType = BACKBONE4;
                }

                if (!HostName || !SwitchName || ifaceNetworkType == NIL_NETWORK) {
                    return nullptr;
                }

                const auto& hostName = *HostName;
                if (host.IsRtc() && !hostName.EndsWith(host.GetName())) {
                    return nullptr;
                }

                const TLine& queue_(host.GetLine());
                const TSwitch& switch_(queue_.CreateSwitch(*SwitchName, ExistedNames));

                return CreateHostInterface(
                    hostName, host, switch_, ifaceNetworkType, Vlan, Vrf,
                    Ipv4.GetOrElse(Default<TIpAddress>()), Ipv6.GetOrElse(Default<TIpAddress>())
                );
            }

            const THostInterface* CreateHostInterface(
                    const TString& hostName, const THost& host, const TSwitch& switch_,
                    ENetworkType networkType, int vlan, const TString* vrf,
                    const TIpAddress& ipv4, const TIpAddress& ipv6) {

                const THostInterface& iface(host.CreateInterface(
                    hostName, switch_, networkType,
                    vlan, vrf, ipv4, ipv6,
                    ExistedNames
                ));

                Indexes.SwitchIdSet.insert(&switch_);
                Indexes.HostInterfaceIdSet.insert(&iface);
                Indexes.HostInterfaceNameSet.insert(&iface);

                return &iface;
            }

        private:
            const TDatacenter& CreateDatacenter(const TString& name) {
                const auto it = Indexes.DatacenterSet.find(name);
                if (it.IsEnd()) {
                    return **Indexes.DatacenterSet.emplace(MakeIntrusive<TDatacenter>(name, name)).first;
                } else {
                    return **it;
                }
            }

            const TSwitch* CreateSwitch() {
                if (!DatacenterName || !QueueName || !SwitchName ||
                    !TTopologySettings::Get()->GetKnownDatacenters().contains(*DatacenterName))
                {
                    return nullptr;
                }

                // https://st.yandex-team.ru/NETMON-609
                if (TTopologySettings::Get()->GetUsePodAsQueue() && PodName && PodName->StartsWith("vlx-")) {
                    DatacenterName = &ExistedNames.Intern(TStringBuf("vlx"));
                }
                
                if (!QueueName->StartsWith(*DatacenterName)) {
                    return nullptr;
                }

                // https://st.yandex-team.ru/NETMON-448
                if (TTopologySettings::Get()->GetUsePodAsQueue() && *DatacenterName == "sas") {
                    TString newDc = (*QueueName == "sas-1") ? "sas1" : "sas2";
                    DatacenterName = &ExistedNames.Intern(newDc);
                }

                // https://st.yandex-team.ru/NETMON-482
                if (TTopologySettings::Get()->GetUsePodAsQueue() && WalleProject == "yp-iss-vla-noclab") {
                    DatacenterName = &ExistedNames.Intern(TStringBuf("noclab"));
                    // hack to avoid artifacts in GUI
                    auto newQueueName = SubstGlobalCopy(*QueueName, "vla", "noclab");
                    QueueName = &ExistedNames.Intern(newQueueName);
                    auto newPodName = SubstGlobalCopy(*PodName, "vla", "noclab");
                    PodName = &ExistedNames.Intern(newPodName);
                }

                const TDatacenter& datacenter(CreateDatacenter(*DatacenterName));

                if (!TTopologySettings::Get()->GetUsePodAsQueue()) {
                    const auto& queueMapping(TTopologySettings::Get()->GetQueueMapping());
                    if (queueMapping.contains(*QueueName)) {
                        const auto& newQueueName(queueMapping.at(*QueueName));
                        if (!Indexes.QueueSet.contains(newQueueName) &&
                            !TTopologySettings::Get()->GetKnownQueuesFilter().Check(newQueueName))
                        {
                            return nullptr;
                        }

                        // create fake old queue
                        const TLine& queue_(datacenter.CreateQueue(*QueueName, ExistedNames));
                        Indexes.QueueIdSet.insert(&queue_);
                        Indexes.QueueSet.insert(&queue_);

                        // change name to mapped
                        QueueName = &ExistedNames.Intern(newQueueName);
                    } else if (!Indexes.QueueSet.contains(*QueueName) &&
                               !TTopologySettings::Get()->GetKnownQueuesFilter().Check(*QueueName)) {
                        return nullptr;
                    }
                }

                const TPod* pod = nullptr;
                if (PodName) {
                    pod = &datacenter.CreatePod(*PodName, ExistedNames);
                }

                const TLine& queue_(datacenter.CreateQueue(*QueueName, ExistedNames));
                const TSwitch& switch_(queue_.CreateSwitch(*SwitchName, ExistedNames, pod));

                Indexes.DatacenterIdSet.insert(&datacenter);
                Indexes.QueueIdSet.insert(&queue_);
                Indexes.QueueSet.insert(&queue_);
                Indexes.SwitchIdSet.insert(&switch_);
                Indexes.SwitchSet.insert(&switch_);

                if (pod) {
                    pod->AddSwitch(&switch_);
                    Indexes.PodIdSet.insert(pod);
                    Indexes.PodSet.insert(pod);
                }

                return &switch_;
            }

            TTopologyIndexes& Indexes;
            TExistedNames& ExistedNames;

            const TString* DatacenterName = nullptr;
            const TString* QueueName = nullptr;
            const TString* PodName = nullptr;
            const TString* SwitchName = nullptr;
            const TString* HostName = nullptr;

            ui64 InventoryNumber = 0;

            TMaybe<EParsedNetworkType> NetworkType;
            int Vlan = 0;
            const TString* Vrf = nullptr;

            TMaybe<TIpAddress> Ipv4;
            TMaybe<TIpAddress> Ipv6;

            TStringSet Owners;
            TMaybe<TString> WalleProject;
            TStringSet WalleTags;
        };
    }

    const TString& TExistedNames::Intern(const TStringBuf& needle) {
        TGuard<TAdaptiveLock> guard(InternLock);
        const auto it = Identifiers.find(needle);
        if (it.IsEnd()) {
            return *Identifiers.emplace(needle).first;
        } else {
            return *it;
        }
    }

    bool TExistedNames::Exists(const TString& name) const {
        TGuard<TAdaptiveLock> guard(InternLock);
        return !Identifiers.find(name).IsEnd();
    }

    THostInterface::THostInterface(const TString& name, const TString& path,
                                   const THost& host, const TSwitch& switch_,
                                   ENetworkType networkType, int vlan, const TString* vrf,
                                   const TIpAddress& ipv4, const TIpAddress& ipv6)
        : TNetworkObject<THostInterface>(name, path)
        , Host(host)
        , Switch(switch_)
        , NetworkType(networkType)
        , FlatInterface(GetReducedId())
        , Comparator(switch_.GetDatacenter().GetReducedId(),
                     switch_.GetLine().GetReducedId(),
                     switch_.GetReducedId(),
                     host.GetReducedId(),
                     GetReducedId())
        , Vlan(vlan)
        , Vrf(vrf)
        , Ipv4(ipv4)
        , Ipv6(ipv6)
    {
    }

    THost::THost(const TString& name, const TString& path, const TSwitch& switch_,
                 ui64 inventoryNumber, const TStringSet& owners,
                 const TMaybe<TString>& walleProject, const TStringSet& walleTags)
        : TNetworkObject<THost>(name, path)
        , Switch(switch_)
        , FlatHost(GetReducedId())
        , InventoryNumber(inventoryNumber)
        , Owners(owners)
        , WalleProject(walleProject)
        , WalleTags(walleTags)
        , HasRtcWalleTag(WalleTags.contains(TStringBuf("rtc")))
        , Comparator(switch_.GetDatacenter().GetReducedId(),
                     switch_.GetLine().GetReducedId(),
                     switch_.GetReducedId(),
                     GetReducedId())
    {
    }

    const THostInterface& THost::CreateInterface(
            const TString& name, const TSwitch& switch_,
            ENetworkType networkType, int vlan, const TString* vrf,
            const TIpAddress& ipv4, const TIpAddress& ipv6,
            TExistedNames& existedNames) const {
        return AddChild<THostInterface>(InterfaceSet, name, existedNames,
                                        switch_, networkType, vlan, vrf, ipv4, ipv6);
    }

    TSwitch::TSwitch(const TString& name, const TString& path, const TLine& queue_, const TPod* pod)
        : TNetworkObject<TSwitch>(name, path)
        , Queue(queue_)
        , Pod(pod)
        , FlatSwitch(GetReducedId())
        , AllHostsAreRtc(true)
        , Comparator(queue_.GetDatacenter().GetReducedId(),
                     queue_.GetReducedId(),
                     GetReducedId())
    {
    }

    const THost& TSwitch::CreateHost(const TString& name, ui64 inventoryNumber, const TStringSet& owners,
                                     const TMaybe<TString>& walleProject, const TStringSet& walleTags,
                                     TExistedNames& existedNames) const {
        const auto& host = AddChild<THost>(HostSet, name, existedNames,
                                           inventoryNumber, owners, walleProject, walleTags);
        if (inventoryNumber) {
            RealHostSet.emplace(&host);
            AllHostsAreRtc = AllHostsAreRtc && host.IsRtc();
        }
        return host;
    }

    TPod::TPod(const TString& name, const TString& path, const TDatacenter& datacenter)
        : TNetworkObject<TPod>(name, path)
        , Datacenter(datacenter)
    {
    }

    const TSwitch* TPod::FindSwitch(const TString& switchName) const {
        return TSwitch::FindByName(SwitchSet, switchName);
    }

    TLine::TLine(const TString& name, const TString& path, const TDatacenter& datacenter)
        : TNetworkObject<TLine>(name, path)
        , Datacenter(datacenter)
        , FlatQueue(GetReducedId())
        , Comparator(datacenter.GetReducedId(), GetReducedId())
    {
    }

    const TSwitch& TLine::CreateSwitch(const TString& name,
                                       TExistedNames& existedNames,
                                       const TPod* pod) const {
        return AddChild<TSwitch>(SwitchSet, name, existedNames, pod);
    }

    const TSwitch* TLine::FindSwitch(const TString& switchName) const {
        return TSwitch::FindByName(SwitchSet, switchName);
    }

    TDatacenter::TDatacenter(const TString& name, const TString& path)
        : TNetworkObject<TDatacenter>(name, path)
        , FlatDatacenter(GetReducedId())
    {
    }

    const TLine& TDatacenter::CreateQueue(const TString& name, TExistedNames& existedNames) const {
        return AddChild<TLine>(QueueSet, name, existedNames);
    }

    const TLine* TDatacenter::FindQueue(const TString& queueName) const {
        return TLine::FindByName(QueueSet, queueName);
    }

    const TPod& TDatacenter::CreatePod(const TString& name, TExistedNames& existedNames) const {
        return AddChild<TPod>(PodSet, name, existedNames);
    }

    const TPod* TDatacenter::FindPod(const TString& queueName) const {
        return TPod::FindByName(PodSet, queueName);
    }

    TTopology::TTopology(TExistedNames& existedNames)
        : ExistedNames(existedNames)
    {
    }

    void TTopology::ParseFile(const TFile& topologyFile, TAtomic& shouldStop) {
        DEBUG_LOG << "Topology parsing started" << Endl;
        TSimpleTimer timer;

        // parse topology database and create hosts from it
        TFileInput fileStream(topologyFile);
        TBufferedZLibDecompress stream(&fileStream);
        msgpack::unpacker unpacker;
        while (true) {
            if (shouldStop) {
                break;
            }

            unpacker.reserve_buffer(CHUNK_SIZE);
            std::size_t readed = stream.Read(unpacker.buffer(), CHUNK_SIZE);
            if (!readed) {
                break;
            }
            unpacker.buffer_consumed(readed);
            msgpack::unpacked result;
            while (unpacker.next(result)) {
                msgpack::object obj(result.get());
                if (obj.type == msgpack::type::MAP) {
                    CreateHostFromWire(obj);
                }
            }
        }

        auto elapsed = timer.Get();

        DEBUG_LOG << "Topology parsing took " << elapsed << Endl;
        TUnistat::Instance().PushSignalUnsafe(ETopologySignals::TopologyParsingTime, elapsed.MilliSeconds());

        INFO_LOG << "Found " << Indexes.HostNameSet.size() << " hosts in topology" << Endl;
        TUnistat::Instance().PushSignalUnsafe(ETopologySignals::TopologyHostsCount, Indexes.HostNameSet.size());

        TVector<TString> queues;
        queues.reserve(Indexes.QueueSet.size());
        for (const auto& queue : Indexes.QueueSet) {
            queues.push_back(queue->GetName());
        }
        Sort(queues);
        INFO_LOG << "Found " << queues.size() << " queues: " << JoinSeq(","sv, queues) << Endl;
    }

    void TTopology::CreateHostInterfaceFromWire(const THost& host, msgpack::object& ptr) {
        TObjectFactory factory(Indexes, ExistedNames);

        for (auto& item : TMapIterator(ptr.via.map)) {
            if (!item.val.is_nil()) {
                factory.ProcessItem(item);
            }
        }

        factory.CreateHostInterface(host);
    }

    void TTopology::CreateHostFromWire(msgpack::object& ptr) {
        TObjectFactory factory(Indexes, ExistedNames);

        msgpack::object* interfaces = nullptr;
        for (auto& item : TMapIterator(ptr.via.map)) {
            if (item.val.is_nil()) {
                continue;
            }
            if (factory.ProcessItem(item)) {
                continue;
            }

            TStringBuf key = item.key.as<TStringBuf>();
            if (key == TStringBuf("interfaces") && item.val.type == msgpack::type::ARRAY) {
                interfaces = &item.val;
            }
        }

        const THost* host(factory.CreateHost());
        if (!host) {
            return;
        }

        if (interfaces != nullptr) {
            // interfaces is a list of maps
            for (auto& item : TListIterator(interfaces->via.array)) {
                if (item.type == msgpack::type::MAP) {
                    CreateHostInterfaceFromWire(*host, item);
                }
            }
        }
    }

    const THostInterface* TTopology::FindHostInterface(const TString& iface) const {
        return THostInterface::FindByName(Indexes.HostInterfaceNameSet, iface);
    }

    const THostInterface* TTopology::FindHostInterface(const NCommon::THostInterface& iface) const {
        return THostInterface::FindById(Indexes.HostInterfaceIdSet, iface);
    }

    const THost* TTopology::FindHost(const TString& host) const {
        return THost::FindByName(Indexes.HostNameSet, host);
    }

    const THost* TTopology::FindHost(const NCommon::THost& host) const {
        return THost::FindById(Indexes.HostIdSet, host);
    }

    const TDatacenter* TTopology::FindDatacenter(const TString& datacenterName) const {
        return TDatacenter::FindByName(Indexes.DatacenterSet, datacenterName);
    }

    const TDatacenter* TTopology::FindDatacenter(const NCommon::TDatacenter& datacenter) const {
        return TDatacenter::FindById(Indexes.DatacenterIdSet, datacenter);
    }

    const TLine* TTopology::FindQueue(const NCommon::TLine& queue_) const {
        return TLine::FindById(Indexes.QueueIdSet, queue_);
    }

    const TLine* TTopology::FindQueue(const TString& queueName) const {
        return TLine::FindByName(Indexes.QueueSet, queueName);
    }

    const TPod* TTopology::FindPod(const TString& podName) const {
        return TPod::FindByName(Indexes.PodSet, podName);
    }

    const TSwitch* TTopology::FindSwitch(const NCommon::TSwitch& switch_) const {
        return TSwitch::FindById(Indexes.SwitchIdSet, switch_);
    }

    const TSwitch* TTopology::FindSwitch(const TString& switchName) const {
        return TSwitch::FindByName(Indexes.SwitchSet, switchName);
    }
}

template <>
void Out<NNetmon::THostInterface>(IOutputStream& stream, TTypeTraits<NNetmon::THostInterface>::TFuncParam obj) {
    stream << "THostInterface(name=" << obj.GetName() << ")";
}

template <>
void Out<NNetmon::THost>(IOutputStream& stream, TTypeTraits<NNetmon::THost>::TFuncParam obj) {
    stream << "THost(name=" << obj.GetName() << ")";
}

template <>
void Out<NNetmon::TSwitch>(IOutputStream& stream, TTypeTraits<NNetmon::TSwitch>::TFuncParam obj) {
    stream << "TSwitch(name=" << obj.GetName() << ")";
}

template <>
void Out<NNetmon::TPod>(IOutputStream& stream, TTypeTraits<NNetmon::TPod>::TFuncParam obj) {
    stream << "TPod(name=" << obj.GetName() << ")";
}

template <>
void Out<NNetmon::TLine>(IOutputStream& stream, TTypeTraits<NNetmon::TLine>::TFuncParam obj) {
    stream << "TLine(name=" << obj.GetName() << ")";
}

template <>
void Out<NNetmon::TDatacenter>(IOutputStream& stream, TTypeTraits<NNetmon::TDatacenter>::TFuncParam obj) {
    stream << "TDatacenter(name=" << obj.GetName() << ")";
}
