#pragma once

#include <infra/netmon/idl/common.fbs.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/msgpack_helpers.h>
#include <infra/netmon/topology/types.h>

#include <util/digest/city.h>
#include <util/generic/algorithm.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/generic/noncopyable.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/system/fstat.h>

namespace NNetmon {
    class THost;
    class TSwitch;
    class TPod;
    class TLine;
    class TDatacenter;
    class TTopology;

    using TStringSet = THashSet<TString>;

    struct TExistedNames {
        TStringSet Identifiers;
        mutable TAdaptiveLock InternLock;

        const TString& Intern(const TStringBuf& needle);
        bool Exists(const TString& name) const;
    };

    struct TDoubleComparator {
        inline TDoubleComparator(ui64 first, ui64 second) noexcept
            : First(first)
            , Second(second)
        {
        }

        inline bool operator<(const TDoubleComparator& other) const noexcept {
            return std::tie(First, Second) < std::tie(other.First, other.Second);
        }

        const ui64 First;
        const ui64 Second;
    };

    struct TTripleComparator {
        inline TTripleComparator(ui64 first, ui64 second, ui64 third) noexcept
            : First(first)
            , Second(second)
            , Third(third)
        {
        }

        inline bool operator<(const TTripleComparator& other) const noexcept {
            return std::tie(First, Second, Third) < std::tie(other.First, other.Second, other.Third);
        }

        const ui64 First;
        const ui64 Second;
        const ui64 Third;
    };

    struct TQuadrupleComparator {
        inline TQuadrupleComparator(ui64 first, ui64 second, ui64 third, ui64 fourth) noexcept
            : First(first)
            , Second(second)
            , Third(third)
            , Fourth(fourth)
        {
        }

        inline bool operator<(const TQuadrupleComparator& other) const noexcept {
            return std::tie(First, Second, Third, Fourth) < std::tie(other.First, other.Second, other.Third, other.Fourth);
        }

        const ui64 First;
        const ui64 Second;
        const ui64 Third;
        const ui64 Fourth;
    };

    struct TQuintupleComparator {
        inline TQuintupleComparator(ui64 first, ui64 second, ui64 third, ui64 fourth, ui64 fifth) noexcept
            : First(first)
            , Second(second)
            , Third(third)
            , Fourth(fourth)
            , Fifth(fifth)
        {
        }

        inline bool operator<(const TQuintupleComparator& other) const noexcept {
            return std::tie(First, Second, Third, Fourth, Fifth) < std::tie(other.First, other.Second, other.Third, other.Fourth, other.Fifth);
        }

        const ui64 First;
        const ui64 Second;
        const ui64 Third;
        const ui64 Fourth;
        const ui64 Fifth;
    };

    template <class T>
    inline bool IsLess(const T& leftTarget, const T& rightTarget,
                       const T& leftSource, const T& rightSource) noexcept {
        if (leftTarget < rightTarget) {
            return true;
        } else if (leftTarget == rightTarget) {
            return leftSource < rightSource;
        } else {
            return false;
        }
    };

    template <>
    inline bool IsLess<TDoubleComparator>(
            const TDoubleComparator& leftTarget, const TDoubleComparator& rightTarget,
            const TDoubleComparator& leftSource, const TDoubleComparator& rightSource) noexcept {
        return std::tie(
            leftTarget.First, leftSource.First, leftTarget.Second, leftSource.Second
        ) < std::tie(
            rightTarget.First, rightSource.First, rightTarget.Second, rightSource.Second
        );
    };

    template <>
    inline bool IsLess<TTripleComparator>(
            const TTripleComparator& leftTarget, const TTripleComparator& rightTarget,
            const TTripleComparator& leftSource, const TTripleComparator& rightSource) noexcept {
        return std::tie(
            leftTarget.First, leftSource.First, leftTarget.Second, leftSource.Second,
            leftTarget.Third, leftSource.Third
        ) < std::tie(
            rightTarget.First, rightSource.First, rightTarget.Second, rightSource.Second,
            rightTarget.Third, rightSource.Third
        );
    };

    template <>
    inline bool IsLess<TQuadrupleComparator>(
            const TQuadrupleComparator& leftTarget, const TQuadrupleComparator& rightTarget,
            const TQuadrupleComparator& leftSource, const TQuadrupleComparator& rightSource) noexcept {
        return std::tie(
            leftTarget.First, leftSource.First, leftTarget.Second, leftSource.Second,
            leftTarget.Third, leftSource.Third, leftTarget.Fourth, leftSource.Fourth
        ) < std::tie(
            rightTarget.First, rightSource.First, rightTarget.Second, rightSource.Second,
            rightTarget.Third, rightSource.Third, rightTarget.Fourth, rightSource.Fourth
        );
    };

    template <>
    inline bool IsLess<TQuintupleComparator>(
            const TQuintupleComparator& leftTarget, const TQuintupleComparator& rightTarget,
            const TQuintupleComparator& leftSource, const TQuintupleComparator& rightSource) noexcept {
        return std::tie(
            leftTarget.First, leftSource.First, leftTarget.Second, leftSource.Second,
            leftTarget.Third, leftSource.Third, leftTarget.Fourth, leftSource.Fourth,
            leftTarget.Fifth, leftSource.Fifth
        ) < std::tie(
            rightTarget.First, rightSource.First, rightTarget.Second, rightSource.Second,
            rightTarget.Third, rightSource.Third, rightTarget.Fourth, rightSource.Fourth,
            rightTarget.Fifth, rightSource.Fifth
        );
    };

    namespace {
        template <class T>
        struct TNameHashFunctor {
            inline size_t operator()(const T* x) const noexcept {
                return x->GetHash();
            }

            inline size_t operator()(const typename T::TRef& x) const noexcept {
                return x->GetHash();
            }

            inline size_t operator()(const TStringBuf& x) const noexcept {
                return ComputeHash(x);
            }

            inline size_t operator()(const TString& x) const noexcept {
                return ComputeHash(x);
            }
        };

        template <class T>
        struct TIdHashFunctor {
            inline size_t operator()(const T* x) const noexcept {
                return x->GetReducedId();
            }

            inline size_t operator()(const typename T::TRef& x) const noexcept {
                return x->GetReducedId();
            }

            inline size_t operator()(const uint128& x) const noexcept {
                return Hash128to64(x);
            }

            inline size_t operator()(const size_t& x) const noexcept {
                return x;
            }
        };

        // compare object with itself or with it's name
        template <class T>
        struct TEqualFunctor {
            inline bool operator()(const T* lhs, const T* rhs) const noexcept {
                return *lhs == *rhs;
            }

            inline bool operator()(const typename T::TRef& lhs, const typename T::TRef& rhs) const noexcept {
                return *lhs == *rhs;
            }

            inline bool operator()(const T* lhs, const TStringBuf& rhs) const noexcept {
                return lhs->GetName() == rhs;
            }

            inline bool operator()(const typename T::TRef& lhs, const TStringBuf& rhs) const noexcept {
                return lhs->GetName() == rhs;
            }

            inline bool operator()(const T* lhs, const TString& rhs) const noexcept {
                return lhs->GetName() == rhs;
            }

            inline bool operator()(const typename T::TRef& lhs, const TString& rhs) const noexcept {
                return lhs->GetName() == rhs;
            }

            inline bool operator()(const T* lhs, const uint128& rhs) const noexcept {
                return lhs->GetId() == rhs;
            }

            inline bool operator()(const typename T::TRef& lhs, const uint128& rhs) const noexcept {
                return lhs->GetId() == rhs;
            }

            inline bool operator()(const T* lhs, const size_t& rhs) const noexcept {
                return lhs->GetReducedId() == rhs;
            }

            inline bool operator()(const typename T::TRef& lhs, const size_t& rhs) const noexcept {
                return lhs->GetReducedId() == rhs;
            }
        };

        template <class T>
        class TNetworkObject : public TAtomicRefCount<T> {
        public:
            using TRef = TIntrusiveConstPtr<T>;

            using TNameSet = THashSet<TRef, TNameHashFunctor<T>, TEqualFunctor<T>>;
            using TIdRefSet = THashSet<const T*, TIdHashFunctor<T>, TEqualFunctor<T>>;
            using TNameRefSet = THashSet<const T*, TNameHashFunctor<T>, TEqualFunctor<T>>;

            static const T* FindByName(const TNameSet& objects, const TString& name) {
                const auto it = objects.find(name);
                if (it.IsEnd()) {
                    return nullptr;
                } else {
                    return it->Get();
                }
            }

            template <class TStruct>
            static const T* FindById(const TIdRefSet& objects, const TStruct& obj) {
                const auto it = objects.find(obj.Id());
                if (it.IsEnd()) {
                    return nullptr;
                } else {
                    return *it;
                }
            }

            template <class TStruct>
            static const T* FindByName(const TNameRefSet& objects, const TStruct& obj) {
                const auto it = objects.find(obj);
                if (it.IsEnd()) {
                    return nullptr;
                } else {
                    return *it;
                }
            }

            TNetworkObject(const TString& name, const TString& path)
                : Name(name)
                , Path(path)
                , Id(CityHash128(Path))
                , ReducedId(Hash128to64(Id))
                , Hash(ComputeHash(name))
            {
            }

            virtual ~TNetworkObject() = default;

            inline const TString& GetName() const noexcept {
                return Name;
            }

            inline const TString& GetPath() const noexcept {
                return Path;
            }

            inline uint128 GetId() const noexcept {
                return Id;
            }

            inline ui64 GetReducedId() const noexcept {
                return ReducedId;
            }

            inline size_t GetHash() const noexcept {
                return Hash;
            }

            inline bool StillUsed() const noexcept {
                // one reference should always exists because of parent object
                return this->RefCount() > 1;
            }

            inline bool operator==(const T& rhs) const noexcept {
                return Id == rhs.Id;
            }

            inline bool operator!=(const T& rhs) const noexcept {
                return !(*this == rhs);
            }

        protected:
            template <class TChild, class... Args>
            const TChild& AddChild(typename TChild::TNameSet& childrenSet,
                                   const TString& name,
                                   TExistedNames& existedNames,
                                   Args&&... args) const {
                const auto it = childrenSet.find(name);
                if (it.IsEnd()) {
                    const TString& childPath = existedNames.Intern(TStringBuilder() << GetPath() << "/" << name);
                    return **childrenSet.emplace(MakeIntrusive<TChild>(
                        name, childPath, static_cast<const T&>(*this),
                        std::forward<Args>(args)...
                    )).first;
                } else {
                    // if child element already exists, ignore the duplicate
                    return **it;
                }
            }

        private:
            const TString& Name;
            const TString& Path;
            const uint128 Id;
            const ui64 ReducedId;
            const size_t Hash;
        };
    }

    class THostInterface: public TNetworkObject<THostInterface> {
    public:
        // name is interface fqdn in our case
        THostInterface(const TString& name, const TString& path,
                       const THost& host, const TSwitch& switch_,
                       ENetworkType networkType, int vlan, const TString* vrf,
                       const TIpAddress& ipv4, const TIpAddress& ipv6);

        inline const THost& GetHost() const noexcept {
            return Host;
        }

        inline const TSwitch& GetSwitch() const noexcept {
            return Switch;
        }
        inline const TLine& GetLine() const noexcept;
        inline const TPod* GetPod() const noexcept;
        inline const TDatacenter& GetDatacenter() const noexcept;
        inline ENetworkType GetNetworkType() const noexcept {
            return NetworkType;
        }
        inline int GetVlan() const noexcept {
            return Vlan;
        }
        inline bool IsMtnVlan() const noexcept {
            return Vlan == EVlans::MTN_BACKBONE || Vlan == EVlans::MTN_FASTBONE;
        }
        inline const TString* GetVrf() const noexcept {
            return Vrf;
        }

        inline const TIpAddress& GetIpv4() const noexcept {
            return Ipv4;
        }
        inline const TIpAddress& GetIpv6() const noexcept {
            return Ipv6;
        }

        inline const NCommon::THostInterface& ToProto() const noexcept {
            return FlatInterface;
        }

        inline const TQuintupleComparator& GetComparator() const noexcept {
            return Comparator;
        }

        inline bool operator<(const THostInterface& other) const noexcept {
            return Comparator < other.Comparator;
        }

    private:
        const THost& Host;
        const TSwitch& Switch;
        const ENetworkType NetworkType;
        const NCommon::THostInterface FlatInterface;
        const TQuintupleComparator Comparator;
        const int Vlan;
        const TString* Vrf;
        const TIpAddress Ipv4;
        const TIpAddress Ipv6;
    };

    class THost: public TNetworkObject<THost> {
    public:
        // name is fqdn in our case
        THost(const TString& name, const TString& path, const TSwitch& switch_,
              ui64 inventoryNumber, const TStringSet& owners,
              const TMaybe<TString>& walleProject, const TStringSet& walleTags);

        inline const TDatacenter& GetDatacenter() const noexcept;
        inline const TLine& GetLine() const noexcept;
        inline const TPod* GetPod() const noexcept;
        inline const TSwitch& GetSwitch() const noexcept;

        inline const THostInterface::TNameSet& GetInterfaces() const noexcept {
            return InterfaceSet;
        }
        const THostInterface& CreateInterface(
                const TString& name, const TSwitch& switch_,
                ENetworkType networkType, int vlan, const TString* vrf,
                const TIpAddress& ipv4, const TIpAddress& ipv6,
                TExistedNames& existedNames) const;

        inline const THostInterface* GetBackboneInterface() const noexcept {
            for (const auto& iface : InterfaceSet) {
                if (iface->GetNetworkType() == ENetworkType::BACKBONE6 &&
                    !iface->IsMtnVlan())
                {
                    return &(*iface);
                }
            }
            return nullptr;
        }
        inline const THostInterface* GetFastboneInterface() const noexcept {
            for (const auto& iface : InterfaceSet) {
                if (iface->GetNetworkType() == ENetworkType::FASTBONE6 &&
                    !iface->IsMtnVlan())
                {
                    return &(*iface);
                }
            }
            return nullptr;
        }
        inline const THostInterface* GetInterfaceByVlan(int vlan) const noexcept {
            for (const auto& iface : InterfaceSet) {
                if (iface->GetVlan() == vlan) {
                    return &(*iface);
                }
            }
            return nullptr;
        }
        inline const THostInterface* GetIpv4Interface() const noexcept {
            for (const auto& iface : InterfaceSet) {
                // hack to check if TIpAddress is initialized
                if (!iface->GetIpv4().ToBytes().Empty()) {
                    return &(*iface);
                }
            }
            return nullptr;
        }

        inline bool IsVirtual() const {
            return !InventoryNumber;
        }
        inline const TStringSet& GetOwners() const {
            return Owners;
        }
        inline const TMaybe<TString>& GetWalleProject() const {
            return WalleProject;
        }
        inline const TStringSet& GetWalleTags() const {
            return WalleTags;
        }
        inline bool IsRtc() const noexcept {
            return HasRtcWalleTag;
        }

        inline const NCommon::THost& ToProto() const noexcept {
            return FlatHost;
        }

        inline const TQuadrupleComparator& GetComparator() const noexcept {
            return Comparator;
        }

        inline bool operator<(const THost& other) const noexcept {
            return Comparator < other.Comparator;
        }

    private:
        const TSwitch& Switch;
        const NCommon::THost FlatHost;
        const ui64 InventoryNumber;
        const TStringSet Owners;
        const TMaybe<TString> WalleProject;
        const TStringSet WalleTags;
        const bool HasRtcWalleTag;
        mutable THostInterface::TNameSet InterfaceSet;
        const TQuadrupleComparator Comparator;
    };

    class TSwitch: public TNetworkObject<TSwitch> {
    public:
        TSwitch(const TString& name, const TString& path, const TLine& queue_, const TPod* pod = nullptr);

        inline const TLine& GetLine() const noexcept {
            return Queue;
        }
        inline const TPod* GetPod() const noexcept {
            return Pod;
        }
        inline const TDatacenter& GetDatacenter() const noexcept;
        inline const THost::TNameSet& GetHosts() const noexcept {
            return HostSet;
        }
        inline const THost::TNameRefSet& GetRealHosts() const noexcept {
            return RealHostSet;
        }

        inline bool IsRtc() const noexcept {
            return AllHostsAreRtc;
        }

        const THost& CreateHost(const TString& name, ui64 inventoryNumber, const TStringSet& owners,
                                const TMaybe<TString>& walleProject, const TStringSet& walleTags,
                                TExistedNames& existedNames) const;

        inline const NCommon::TSwitch& ToProto() const noexcept {
            return FlatSwitch;
        }

        inline const TTripleComparator& GetComparator() const noexcept {
            return Comparator;
        }

        inline bool operator<(const TSwitch& other) const noexcept {
            return Comparator < other.Comparator;
        }

    private:
        const TLine& Queue;
        const TPod* Pod; // may be null
        const NCommon::TSwitch FlatSwitch;
        mutable THost::TNameSet HostSet;
        mutable THost::TNameRefSet RealHostSet;
        mutable bool AllHostsAreRtc;
        const TTripleComparator Comparator;
    };

    class TPod: public TNetworkObject<TPod> {
    public:
        TPod(const TString& name, const TString& path, const TDatacenter& datacenter);

        inline const TDatacenter& GetDatacenter() const noexcept {
            return Datacenter;
        }
        inline const TSwitch::TNameRefSet& GetSwitches() const noexcept {
            return SwitchSet;
        }

        const TSwitch* FindSwitch(const TString& switchName) const;
        inline void AddSwitch(const TSwitch* switch_) const {
            SwitchSet.emplace(switch_);
        }

    private:
        const TDatacenter& Datacenter;
        mutable TSwitch::TNameRefSet SwitchSet;
    };

    class TLine: public TNetworkObject<TLine> {
    public:
        TLine(const TString& name, const TString& path, const TDatacenter& datacenter);

        inline const TDatacenter& GetDatacenter() const noexcept {
            return Datacenter;
        }
        inline const TSwitch::TNameSet& GetSwitches() const noexcept {
            return SwitchSet;
        }

        const TSwitch* FindSwitch(const TString& switchName) const;
        const TSwitch& CreateSwitch(const TString& name,
                                    TExistedNames& existedNames,
                                    const TPod* pod = nullptr) const;

        inline const NCommon::TLine& ToProto() const noexcept {
            return FlatQueue;
        }

        inline const TDoubleComparator& GetComparator() const noexcept {
            return Comparator;
        }

        inline bool operator<(const TLine& other) const noexcept {
            return Comparator < other.Comparator;
        }

        inline bool operator<(const TSwitch& rhs) const noexcept {
            return *this < rhs.GetLine();
        }

    private:
        const TDatacenter& Datacenter;
        const NCommon::TLine FlatQueue;
        mutable TSwitch::TNameSet SwitchSet;
        const TDoubleComparator Comparator;
    };

    class TDatacenter: public TNetworkObject<TDatacenter> {
    public:
        TDatacenter(const TString& name, const TString& path);

        inline const TLine::TNameSet& GetLines() const {
            return QueueSet;
        }
        inline const TPod::TNameSet& GetPods() const {
            return PodSet;
        }

        const TLine& CreateQueue(const TString& name, TExistedNames& existedNames) const;
        const TLine* FindQueue(const TString& queueName) const;
        const TPod& CreatePod(const TString& name, TExistedNames& existedNames) const;
        const TPod* FindPod(const TString& queueName) const;

        inline const NCommon::TDatacenter& ToProto() const noexcept {
            return FlatDatacenter;
        }

        inline bool operator<(const TDatacenter& rhs) const noexcept {
            return GetReducedId() < rhs.GetReducedId();
        }

        inline bool operator<(const TSwitch& rhs) const noexcept {
            return *this < rhs.GetDatacenter();
        }

    private:
        const NCommon::TDatacenter FlatDatacenter;
        mutable TLine::TNameSet QueueSet;
        mutable TPod::TNameSet PodSet;
    };

    inline const TLine& THostInterface::GetLine() const noexcept {
        return Switch.GetLine();
    }

    inline const TPod* THostInterface::GetPod() const noexcept {
        return Switch.GetPod();
    }

    inline const TDatacenter& THostInterface::GetDatacenter() const noexcept {
        return Switch.GetLine().GetDatacenter();
    }

    inline const TDatacenter& THost::GetDatacenter() const noexcept {
        return Switch.GetLine().GetDatacenter();
    }

    inline const TLine& THost::GetLine() const noexcept {
        return Switch.GetLine();
    }

    inline const TPod* THost::GetPod() const noexcept {
        return Switch.GetPod();
    }

    inline const TSwitch& THost::GetSwitch() const noexcept {
        return Switch;
    }

    inline const TDatacenter& TSwitch::GetDatacenter() const noexcept {
        return Queue.GetDatacenter();
    }

    struct TTopologyIndexes: public TNonCopyable {
        TDatacenter::TNameSet DatacenterSet;
        TDatacenter::TIdRefSet DatacenterIdSet;
        TLine::TIdRefSet QueueIdSet;
        TLine::TNameRefSet QueueSet;
        TPod::TIdRefSet PodIdSet;
        TPod::TNameRefSet PodSet;
        TSwitch::TIdRefSet SwitchIdSet;
        TSwitch::TNameRefSet SwitchSet;
        THost::TIdRefSet HostIdSet;
        THost::TNameRefSet HostNameSet;
        THostInterface::TIdRefSet HostInterfaceIdSet;
        THostInterface::TNameRefSet HostInterfaceNameSet;
    };

    class TTopology: public TNonCopyable, public TAtomicRefCount<TTopology> {
    public:
        using TBox = TIntrusiveLockedBox<TTopology>;
        using TRef = TBox::TConstValueRef;

        template <class T>
        class TObjectRef: public TPointerBase<TObjectRef<T>, const T> {
        public:
            using TType = T;

            explicit inline TObjectRef() noexcept
            {
            }
            explicit inline TObjectRef(const T& obj) noexcept
                : Object(&const_cast<T&>(obj))
            {
            }
            explicit inline TObjectRef(const T* obj) noexcept
                : Object(obj != nullptr ? &const_cast<T&>(*obj) : nullptr)
            {
            }
            explicit inline TObjectRef(const TObjectRef& ref) noexcept
                : Object(ref.Object)
            {
            }

            inline const T* Get() const noexcept {
                return Object.Get();
            }

            inline size_t GetHash() const noexcept {
                return Object->GetHash();
            }

            inline bool operator<(const T& rhs) const {
                return this->Object != nullptr ? (*this->Object < rhs) : false;
            }
            inline bool operator==(const T& rhs) const {
                return this->Object != nullptr ? (*this->Object == rhs) : false;
            }

            inline bool operator<(const TObjectRef<T>& rhs) const {
                if (this->Object == nullptr || rhs.Object == nullptr) {
                    return false;
                } else {
                    return *this->Object < *rhs.Object;
                }
            }
            inline bool operator==(const TObjectRef<T>& rhs) const {
                if (this->Object == nullptr || rhs.Object == nullptr) {
                    return false;
                } else {
                    return *this->Object == *rhs.Object;
                }
            }

        private:
            const typename T::TRef Object;
        };

        class TDatacenterRef: public TObjectRef<TDatacenter> {
        public:
            using TObjectRef::TObjectRef;
        };

        class TLineRef: public TObjectRef<TLine> {
        public:
            using TObjectRef::TObjectRef;

            TDatacenterRef GetDatacenter() const noexcept {
                return TDatacenterRef(Get()->GetDatacenter());
            }
        };

        class TPodRef : public TObjectRef<TPod> {
        public:
            using TObjectRef::TObjectRef;

            TDatacenterRef GetDatacenter() const noexcept {
                return TDatacenterRef(Get()->GetDatacenter());
            }
        };

        class TSwitchRef: public TObjectRef<TSwitch> {
        public:
            using TObjectRef::TObjectRef;

            TPodRef GetPod() const noexcept {
                return TPodRef(Get()->GetPod());
            }

            TLineRef GetLine() const noexcept {
                return TLineRef(Get()->GetLine());
            }

            TDatacenterRef GetDatacenter() const noexcept {
                return TDatacenterRef(Get()->GetDatacenter());
            }
        };

        class THostRef: public TObjectRef<THost> {
        public:
            using TObjectRef::TObjectRef;

            TSwitchRef GetSwitch() const noexcept {
                return TSwitchRef(Get()->GetSwitch());
            }

            TPodRef GetPod() const noexcept {
                return TPodRef(Get()->GetPod());
            }

            TLineRef GetLine() const noexcept {
                return TLineRef(Get()->GetLine());
            }

            TDatacenterRef GetDatacenter() const noexcept {
                return TDatacenterRef(Get()->GetDatacenter());
            }
        };

        class THostInterfaceRef: public TObjectRef<THostInterface> {
        public:
            using TObjectRef::TObjectRef;

            THostRef GetHost() const noexcept {
                return THostRef(Get()->GetHost());
            }

            TSwitchRef GetSwitch() const noexcept {
                return TSwitchRef(Get()->GetSwitch());
            }

            TPodRef GetPod() const noexcept {
                return TPodRef(Get()->GetPod());
            }

            TLineRef GetLine() const noexcept {
                return TLineRef(Get()->GetLine());
            }

            TDatacenterRef GetDatacenter() const noexcept {
                return TDatacenterRef(Get()->GetDatacenter());
            }
        };

        TTopology(TExistedNames& existedNames);

        void ParseFile(const TFile& topologyFile, TAtomic& shouldStop);

        const THostInterface* FindHostInterface(const TString& iface) const;
        const THostInterface* FindHostInterface(const NCommon::THostInterface& iface) const;

        const THost* FindHost(const TString& host) const;
        const THost* FindHost(const NCommon::THost& host) const;

        const TDatacenter* FindDatacenter(const TString& datacenterName) const;
        const TDatacenter* FindDatacenter(const NCommon::TDatacenter& datacenter) const;

        const TLine* FindQueue(const TString& queueName) const;
        const TLine* FindQueue(const NCommon::TLine& queue_) const;

        const TPod* FindPod(const TString& podName) const;

        const TSwitch* FindSwitch(const TString& switchName) const;
        const TSwitch* FindSwitch(const NCommon::TSwitch& switch_) const;

        inline const TDatacenter::TNameSet& GetDatacenters() const {
            return Indexes.DatacenterSet;
        }

        template <class F>
        inline void ForEachSwitch(F&& f) const {
            for (const auto& switch_ : Indexes.SwitchSet) {
                f(*switch_);
            }
        }

        template <class F>
        inline void ForEachHost(F&& f) const {
            for (const auto& host : Indexes.HostIdSet) {
                f(*host);
            }
        }

        template <class F>
        inline void ForEachInterface(F&& f) const {
            for (const auto& iface : Indexes.HostInterfaceIdSet) {
                f(*iface);
            }
        }

    private:
        void CreateHostInterfaceFromWire(const THost& host, msgpack::object& ptr);
        void CreateHostFromWire(msgpack::object& ptr);

        TExistedNames& ExistedNames;
        TTopologyIndexes Indexes;
    };
}

namespace {
    template <class TRef>
    struct TObjectRefHash {
        inline size_t operator()(const TRef& key) const {
            return key.GetHash();
        }
        inline size_t operator()(const typename TRef::TType& key) const {
            return key.GetHash();
        }
    };

    template <class TRef>
    struct TObjectRefEqualTo {
        inline bool operator()(const TRef& lhs, const TRef& rhs) const {
            return lhs == rhs;
        }
        inline bool operator()(const TRef& lhs, const typename TRef::TType& rhs) const {
            return lhs == rhs;
        }
    };
}

template<>
struct THash<NNetmon::TTopology::THostInterfaceRef>: TObjectRefHash<NNetmon::TTopology::THostInterfaceRef> {
};

template<>
struct THash<NNetmon::TTopology::THostRef>: TObjectRefHash<NNetmon::TTopology::THostRef> {
};

template<>
struct THash<NNetmon::TTopology::TSwitchRef>: TObjectRefHash<NNetmon::TTopology::TSwitchRef> {
};

template<>
struct THash<NNetmon::TTopology::TPodRef>: TObjectRefHash<NNetmon::TTopology::TPodRef> {
};

template<>
struct THash<NNetmon::TTopology::TLineRef>: TObjectRefHash<NNetmon::TTopology::TLineRef> {
};

template<>
struct THash<NNetmon::TTopology::TDatacenterRef>: TObjectRefHash<NNetmon::TTopology::TDatacenterRef> {
};

template <>
struct TEqualTo<NNetmon::TTopology::THostInterfaceRef>: TObjectRefEqualTo<NNetmon::TTopology::THostInterfaceRef> {
};

template<>
struct TEqualTo<NNetmon::TTopology::THostRef>: TObjectRefEqualTo<NNetmon::TTopology::THostRef> {
};

template<>
struct TEqualTo<NNetmon::TTopology::TSwitchRef>: TObjectRefEqualTo<NNetmon::TTopology::TSwitchRef> {
};

template<>
struct TEqualTo<NNetmon::TTopology::TPodRef>: TObjectRefEqualTo<NNetmon::TTopology::TPodRef> {
};
