#include <infra/netmon/topology/topology_selector.h>
#include <infra/netmon/topology/settings.h>

namespace NNetmon {
    namespace {
        using THostInterfaceSet = THostInterface::TIdRefSet;

        using EDirection = TExpressionClause::EDirection;

        using TInterfaceWithDirection = std::pair<const THostInterface*, EDirection>;
        using TInterfaceWithDirectionList = TVector<TInterfaceWithDirection>;

        using TInterfaceToExpressionMap = THashMap<TTopology::THostInterfaceRef, TExpressionIdList>;
        using THostToExpressionMap = THashMap<TTopology::THostRef, TExpressionIdList>;

        struct TExpressionObjects {
            TTopologySelector::THostSet Hosts;
            TTopologySelector::TSwitchSet Switches;
            TTopologySelector::TPodSet Pods;
            TTopologySelector::TLineSet Lines;
            TTopologySelector::TDatacenterSet Datacenters;

            struct THostAccessor {
                using TObjectSet = TTopologySelector::THostSet;

                static const TObjectSet& Access(const TExpressionObjects& objects) {
                    return objects.Hosts;
                }
            };

            struct TSwitchAccessor {
                using TObjectSet = TTopologySelector::TSwitchSet;

                static const TObjectSet& Access(const TExpressionObjects& objects) {
                    return objects.Switches;
                }
            };

            struct TPodAccessor {
                using TObjectSet = TTopologySelector::TPodSet;

                static const TObjectSet& Access(const TExpressionObjects& objects) {
                    return objects.Pods;
                }
            };

            struct TLineAccessor {
                using TObjectSet = TTopologySelector::TLineSet;

                static const TObjectSet& Access(const TExpressionObjects& objects) {
                    return objects.Lines;
                }
            };

            struct TDatacenterAccessor {
                using TObjectSet = TTopologySelector::TDatacenterSet;

                static const TObjectSet& Access(const TExpressionObjects& objects) {
                    return objects.Datacenters;
                }
            };
        };
        using TExpressionToObjectsMap = THashMap<TExpressionId, TExpressionObjects>;

        static const TExpressionDnf DEFAULT_EXPRESSION(ParseExpression("virtual=false"));
        static const TExpressionId DEFAULT_EXPRESSION_ID(DEFAULT_EXPRESSION.hash());

        struct THostIndexes : public TNonCopyable {
            THostIndexes(TTopology::TBox::TConstValueRef topology,
                         const TGroupStorage::TExistingHosts& hostsInGroups)
            {
                topology->ForEachInterface([&](const THostInterface& iface) {
                    AllInterfaces.emplace(&iface);

                    InterfacesIsVirtual[iface.GetHost().IsVirtual()].emplace(&iface);

                    InterfacesInDatacenter[iface.GetDatacenter().GetName()].emplace(&iface);
                    InterfacesInQueue[iface.GetLine().GetName()].emplace(&iface);
                    InterfacesInSwitch[iface.GetSwitch().GetName()].emplace(&iface);

                    if (iface.GetVlan()) {
                        InterfacesInVlan[iface.GetVlan()].emplace(&iface);
                    }
                    if (iface.GetVrf()) {
                        InterfacesInVrf[*iface.GetVrf()].emplace(&iface);
                    }

                    if (iface.GetHost().GetWalleProject().Defined()) {
                        InterfacesInWalleProject[iface.GetHost().GetWalleProject().GetRef()].emplace(&iface);
                    }
                    for (const auto& tag : iface.GetHost().GetWalleTags()) {
                        InterfacesInWalleTag[tag].emplace(&iface);
                    }

                    for (const auto& group : hostsInGroups) {
                        if (group.second->find(iface.GetHost().GetName()) != group.second->end()) {
                            if (group.first.Type == SKYNET_GROUP_TYPE) {
                                InterfacesInGroup[group.first.Name].emplace(&iface);
                            } else if (group.first.Type == NANNY_GROUP_TYPE) {
                                InterfacesInNanny[group.first.Name].emplace(&iface);
                            } else if (group.first.Type == GENCFG_GROUP_TYPE) {
                                InterfacesInGencfg[group.first.Name].emplace(&iface);
                            }
                        }
                    }
                });
            }

            THostInterfaceSet AllInterfaces;

            THashMap<bool, THostInterfaceSet> InterfacesIsVirtual;
            THashMap<int, THostInterfaceSet> InterfacesInVlan;
            THashMap<TString, THostInterfaceSet> InterfacesInVrf;
            THashMap<TString, THostInterfaceSet> InterfacesInGroup;
            THashMap<TString, THostInterfaceSet> InterfacesInNanny;
            THashMap<TString, THostInterfaceSet> InterfacesInGencfg;
            THashMap<TString, THostInterfaceSet> InterfacesInWalleProject;
            THashMap<TString, THostInterfaceSet> InterfacesInWalleTag;
            THashMap<TString, THostInterfaceSet> InterfacesInDatacenter;
            THashMap<TString, THostInterfaceSet> InterfacesInQueue;
            THashMap<TString, THostInterfaceSet> InterfacesInSwitch;
        };

        class TExpressionApplicant : public TNonCopyable {
        public:
            TExpressionApplicant(const THostIndexes& indexes,
                                 const TExpressionClause& clause)
                : Indexes(indexes)
                , Clause(clause)
            {
                FindThenIntersect(Indexes.InterfacesInGroup, Clause.PositiveGroupFilter());
                FindThenIntersect(Indexes.InterfacesInNanny, Clause.PositiveNannyFilter());
                FindThenIntersect(Indexes.InterfacesInGencfg, Clause.PositiveGencfgFilter());
                FindThenIntersect(Indexes.InterfacesInWalleProject, Clause.PositiveWalleProjectFilter());
                FindThenIntersect(Indexes.InterfacesInWalleTag, Clause.PositiveWalleTagFilter());
                FindThenIntersect(Indexes.InterfacesInVrf, Clause.PositiveVrfFilter());
                FindThenIntersect(Indexes.InterfacesInVlan, Clause.PositiveVlanFilter());
                FindThenIntersect(Indexes.InterfacesInDatacenter, Clause.PositiveDatacenterFilter());
                FindThenIntersect(Indexes.InterfacesInQueue, Clause.PositiveQueueFilter());
                FindThenIntersect(Indexes.InterfacesInSwitch, Clause.PositiveSwitchFilter());
                FindThenIntersect(Indexes.InterfacesIsVirtual, Clause.PositiveVirtualFilter());

                if (!ResultingSet.Defined()) {
                    ResultingSet.ConstructInPlace(Indexes.AllInterfaces);
                }

                FindThenDifference(Indexes.InterfacesInGroup, Clause.NegativeGroupFilter());
                FindThenDifference(Indexes.InterfacesInNanny, Clause.NegativeNannyFilter());
                FindThenDifference(Indexes.InterfacesInGencfg, Clause.NegativeGencfgFilter());
                FindThenDifference(Indexes.InterfacesInWalleProject, Clause.NegativeWalleProjectFilter());
                FindThenDifference(Indexes.InterfacesInWalleTag, Clause.NegativeWalleTagFilter());
                FindThenDifference(Indexes.InterfacesInVrf, Clause.NegativeVrfFilter());
                FindThenDifference(Indexes.InterfacesInVlan, Clause.NegativeVlanFilter());
                FindThenDifference(Indexes.InterfacesInDatacenter, Clause.NegativeDatacenterFilter());
                FindThenDifference(Indexes.InterfacesInQueue, Clause.NegativeQueueFilter());
                FindThenDifference(Indexes.InterfacesInSwitch, Clause.NegativeSwitchFilter());
                FindThenDifference(Indexes.InterfacesIsVirtual, Clause.NegativeVirtualFilter());
            }

            inline const THostInterfaceSet& GetResultingSet() const noexcept {
                return ResultingSet.GetRef();
            }

            inline EDirection GetDirection() const noexcept {
                if (Clause.DirectionFilter().Empty()) {
                    return EDirection::Both;
                } else {
                    return Clause.DirectionFilter().GetRef();
                }
            }

        private:
            template <class I>
            void Intersect(const I it) {
                if (!it.IsEnd() && ResultingSet.Defined()) {
                    THostInterfaceSet result;
                    for (const auto& host : it->second) {
                        if (ResultingSet->find(host) != ResultingSet->end()) {
                            result.emplace(host);
                        }
                    }
                    ResultingSet->swap(result);
                } else if (!it.IsEnd() && !ResultingSet.Defined()) {
                    ResultingSet.ConstructInPlace(it->second);
                } else if (ResultingSet.Defined()) {
                    ResultingSet->clear();
                } else {
                    ResultingSet.ConstructInPlace();
                }
            }

            template <class T>
            void FindThenIntersect(const THashMap<T, THostInterfaceSet>& container, const TVector<T>& values) {
                for (const auto& element : values) {
                    Intersect(container.find(element));
                }
            }

            template <class I>
            void Difference(const I it) {
                if (!it.IsEnd() && ResultingSet.Defined()) {
                    THostInterfaceSet result;
                    for (const auto& host : ResultingSet.GetRef()) {
                        if (it->second.find(host) == it->second.end()) {
                            result.emplace(host);
                        }
                    }
                    ResultingSet->swap(result);
                }
            }

            template <class T>
            void FindThenDifference(const THashMap<T, THostInterfaceSet>& container, const TVector<T>& values) {
                for (const auto& element : values) {
                    Difference(container.find(element));
                }
            }

            const THostIndexes& Indexes;
            const TExpressionClause& Clause;

            TMaybe<THostInterfaceSet> ResultingSet;
        };

        TInterfaceWithDirectionList FindHostsForExpression(const THostIndexes& indexes, const TExpressionDnf& expression) {
            TInterfaceWithDirectionList result;
            for (const auto& clause : expression) {
                TExpressionApplicant applicant(indexes, clause);
                for (const auto& iface : applicant.GetResultingSet()) {
                    result.emplace_back(iface, applicant.GetDirection());
                }
            }
            return result;
        };
    }

    TSet<TGroupKey> ExtractGroups(const TExpressionDnf& expr) {
        TSet<TGroupKey> groups;
        for (const auto& clause : expr) {
            for (const auto& group : clause.PositiveGroupFilter()) {
                groups.emplace(TGroupKey::Skynet(group));
            }
            for (const auto& group : clause.NegativeGroupFilter()) {
                groups.emplace(TGroupKey::Skynet(group));
            }
            for (const auto& group : clause.PositiveNannyFilter()) {
                groups.emplace(TGroupKey::Nanny(group));
            }
            for (const auto& group : clause.NegativeNannyFilter()) {
                groups.emplace(TGroupKey::Nanny(group));
            }
            for (const auto& group : clause.PositiveGencfgFilter()) {
                groups.emplace(TGroupKey::Gencfg(group));
            }
            for (const auto& group : clause.NegativeGencfgFilter()) {
                groups.emplace(TGroupKey::Gencfg(group));
            }
        }
        return groups;
    }

    class TTopologySelector::TImpl {
    public:
        inline TImpl(TTopology::TBox::TConstValueRef topology,
                     const TVector<TExpressionDnf>& requestedExpressions,
                     const TGroupStorage::TExistingHosts& hostsInGroups)
            : Timestamp(TInstant::Now())
            , Topology(topology)
            , Indexes(Topology, hostsInGroups)
        {
            bool defaultExpressionExists(false);
            for (const auto& expr : requestedExpressions) {
                const auto expressionId(ExpandExpression(expr));
                defaultExpressionExists = defaultExpressionExists || (expressionId == DEFAULT_EXPRESSION_ID);

                const auto extracted(ExtractGroups(expr));
                Groups.insert(extracted.begin(), extracted.end());
            }
            if (!defaultExpressionExists) {
                ExpandExpression(DEFAULT_EXPRESSION);
            }

            for (auto& pair : SourceInterfaceToExpressions) {
                SortExpressions(pair.second);
            }
            for (auto& pair : TargetInterfaceToExpressions) {
                SortExpressions(pair.second);
            }
            for (auto& pair : SourceHostToExpressions) {
                SortExpressions(pair.second);
            }
            for (auto& pair : TargetHostToExpressions) {
                SortExpressions(pair.second);
            }

            for (auto& pair : SourceInterfaceToExpressions) {
                auto& switchWeights(SwitchToWeight[pair.first.GetSwitch()]);
                switchWeights.reserve(requestedExpressions.size());
                for (const auto& expressionId : pair.second) {
                    auto it(switchWeights.find(expressionId));
                    if (it == switchWeights.end()) {
                        switchWeights.emplace(expressionId, 1);
                    } else {
                        it->second++;
                    }
                }
            }
        }

        inline const TExpressionIdList& GetAllExpressions() const noexcept {
            return Expressions;
        }

        inline const TExpressionIdList* FindSourceExpressions(const THostInterface& iface) const noexcept {
            auto it = SourceInterfaceToExpressions.find(iface);
            return !it.IsEnd() ? &it->second : nullptr;
        }

        inline const TExpressionIdList* FindSourceExpressions(const THost& host) const noexcept {
            auto it = SourceHostToExpressions.find(host);
            return !it.IsEnd() ? &it->second : nullptr;
        }

        inline const TExpressionIdList* FindTargetExpressions(const THostInterface& iface) const noexcept {
            auto it = TargetInterfaceToExpressions.find(iface);
            return !it.IsEnd() ? &it->second : nullptr;
        }

        inline const TExpressionIdList* FindTargetExpressions(const THost& host) const noexcept {
            auto it = TargetHostToExpressions.find(host);
            return !it.IsEnd() ? &it->second : nullptr;
        }

        inline ui32 GetWeight(const TSwitch& switch_, TExpressionId expressionId) const noexcept {
            if (!TTopologySettings::Get()->GetConstantSwitchWeights()) {
                auto switchIt(SwitchToWeight.find(switch_));
                if (switchIt != SwitchToWeight.end()) {
                    auto expressionIt(switchIt->second.find(expressionId));
                    if (expressionIt != switchIt->second.end()) {
                        return expressionIt->second;
                    }
                }
            }
            return 1;
        }

        inline size_t CountHosts(TExpressionId expressionId, const IHostFilter& filter) noexcept {
            return FindHosts(expressionId, filter);
        }

        inline THostSet GetHosts(TExpressionId expressionId, const IHostFilter& filter) noexcept {
            THostSet result;
            FindHosts(expressionId, filter, &result);
            return result;
        }

        inline THostSet GetHosts(const TExpressionDnf& expression, const IHostFilter& filter) noexcept {
            THostSet result;
            if (FindHosts(expression.hash(), filter, &result) == 0) {
                for (const auto& pair : FindHostsForExpression(Indexes, expression)) {
                    const auto host(TTopology::THostRef(pair.first->GetHost()));
                    if (filter(host)) {
                        result.emplace(host);
                    }
                }
            }
            return result;
        }

        template <typename TAccessor>
        inline auto GetObjects(TExpressionId expressionId) noexcept {
            typename TAccessor::TObjectSet result;
            const auto found(ExpressionToObjects.find(expressionId));
            if (!found.IsEnd()) {
                const auto& objects(TAccessor::Access(found->second));
                result.insert(objects.cbegin(), objects.cend());
            }
            return result;
        }

        inline const TSet<TGroupKey>& GetGroups() const noexcept {
            return Groups;
        }

        TInstant Created() const noexcept {
            return Timestamp;
        }

    private:
        using TWeightMap = THashMap<TExpressionId, ui32>;

        TExpressionId ExpandExpression(const TExpressionDnf& expr) {
            const auto expressionId(expr.hash());
            Expressions.emplace_back(expressionId);
            for (const auto& pair : FindHostsForExpression(Indexes, expr)) {
                if (pair.second == EDirection::Source || pair.second == EDirection::Both) {
                    MaintainExpression(SourceInterfaceToExpressions, SourceHostToExpressions, expressionId, *pair.first);
                }
                if (pair.second == EDirection::Target || pair.second == EDirection::Both) {
                    MaintainExpression(TargetInterfaceToExpressions, TargetHostToExpressions, expressionId, *pair.first);
                }
            }
            return expressionId;
        }

        void MaintainExpression(TInterfaceToExpressionMap& interfaceToExpressions, THostToExpressionMap& hostToExpressions,
                                TExpressionId expressionId, const THostInterface& iface) {
            auto& ifaceExpressions(interfaceToExpressions[iface]);
            ifaceExpressions.push_back(expressionId);
            PushHeap(ifaceExpressions.begin(), ifaceExpressions.end());

            const auto& host(iface.GetHost());
            auto& hostExpressions(hostToExpressions[host]);
            hostExpressions.push_back(expressionId);
            PushHeap(hostExpressions.begin(), hostExpressions.end());

            auto& expressionObjects(ExpressionToObjects[expressionId]);
            expressionObjects.Hosts.emplace(host);
            expressionObjects.Switches.emplace(host.GetSwitch());
            expressionObjects.Lines.emplace(host.GetLine());
            expressionObjects.Datacenters.emplace(host.GetDatacenter());
            if (host.GetPod()) {
                expressionObjects.Pods.emplace(*host.GetPod());
            }
        }

        void SortExpressions(TExpressionIdList& expressions) const {
            SortHeap(expressions.begin(), expressions.end());
            auto it = Unique(expressions.begin(), expressions.end());
            expressions.erase(it, expressions.end());
        }

        inline size_t FindHosts(TExpressionId expressionId, const IHostFilter& filter, THostSet* resultHosts=nullptr) const {
            size_t hostCount(0);
            const auto found(ExpressionToObjects.find(expressionId));
            if (!found.IsEnd()) {
                const auto& hosts(found->second.Hosts);
                if (filter.IsTrivial()) {
                    if (resultHosts) {
                        resultHosts->insert(hosts.cbegin(), hosts.cend());
                    }
                    return hosts.size();
                }
                for (auto it(hosts.cbegin()); it != hosts.cend(); ++it) {
                    if (filter(*it)) {
                        ++hostCount;
                        if (resultHosts) {
                            resultHosts->emplace(*it);
                        }
                    }
                }
            }
            return hostCount;
        }

        const TInstant Timestamp;

        TTopology::TBox::TConstValueRef Topology;

        const THostIndexes Indexes;

        TExpressionIdList Expressions;

        TInterfaceToExpressionMap SourceInterfaceToExpressions;
        TInterfaceToExpressionMap TargetInterfaceToExpressions;
        THostToExpressionMap SourceHostToExpressions;
        THostToExpressionMap TargetHostToExpressions;

        TExpressionToObjectsMap ExpressionToObjects;

        THashMap<TTopology::TSwitchRef, TWeightMap> SwitchToWeight;

        TSet<TGroupKey> Groups;
    };

    TTopologySelector::TTopologySelector(TTopology::TBox::TConstValueRef topology,
                                         const TVector<TExpressionDnf>& expressions,
                                         const TGroupStorage::TExistingHosts& hostsInGroups)
        : Impl(MakeHolder<TImpl>(topology, expressions, hostsInGroups))
    {
    }

    TTopologySelector::~TTopologySelector()
    {
    }

    const TExpressionIdList& TTopologySelector::GetAllExpressions() const noexcept {
        return Impl->GetAllExpressions();
    }

    const TExpressionIdList* TTopologySelector::FindSourceExpressions(const THostInterface& iface) const noexcept {
        return Impl->FindSourceExpressions(iface);
    }

    const TExpressionIdList* TTopologySelector::FindSourceExpressions(const TTopology::THostInterfaceRef& iface) const noexcept {
        return iface ? Impl->FindSourceExpressions(*iface.Get()) : nullptr;
    }

    const TExpressionIdList* TTopologySelector::FindSourceExpressions(const THost& host) const noexcept {
        return Impl->FindSourceExpressions(host);
    }

    const TExpressionIdList* TTopologySelector::FindSourceExpressions(const TTopology::THostRef& host) const noexcept {
        return host ? Impl->FindSourceExpressions(*host.Get()) : nullptr;
    }

    const TExpressionIdList* TTopologySelector::FindTargetExpressions(const THostInterface& iface) const noexcept {
        return Impl->FindTargetExpressions(iface);
    }

    const TExpressionIdList* TTopologySelector::FindTargetExpressions(const TTopology::THostInterfaceRef& iface) const noexcept {
        return iface ? Impl->FindTargetExpressions(*iface.Get()) : nullptr;
    }

    const TExpressionIdList* TTopologySelector::FindTargetExpressions(const THost& host) const noexcept {
        return Impl->FindTargetExpressions(host);
    }

    const TExpressionIdList* TTopologySelector::FindTargetExpressions(const TTopology::THostRef& host) const noexcept {
        return host ? Impl->FindTargetExpressions(*host.Get()) : nullptr;
    }

    ui32 TTopologySelector::GetWeight(const TSwitch& switch_, TExpressionId expressionId) const noexcept {
        return Impl->GetWeight(switch_, expressionId);
    }

    ui32 TTopologySelector::GetWeight(const TTopology::TSwitchRef& switch_, TExpressionId expressionId) const noexcept {
        return switch_ ? Impl->GetWeight(*switch_.Get(), expressionId) : 1;
    }

    size_t TTopologySelector::CountHosts(TExpressionId expressionId, const IHostFilter& filter) const noexcept {
        return Impl->CountHosts(expressionId, filter);
    }

    TTopologySelector::THostSet TTopologySelector::GetHosts(TExpressionId expressionId, const IHostFilter& filter) const noexcept {
        return Impl->GetHosts(expressionId, filter);
    }

    TTopologySelector::THostSet TTopologySelector::GetHosts(const TExpressionDnf& expression, const IHostFilter& filter) const noexcept {
        return Impl->GetHosts(expression, filter);
    }

    TTopologySelector::TSwitchSet TTopologySelector::GetSwitches(TExpressionId expressionId) const noexcept {
        return Impl->GetObjects<TExpressionObjects::TSwitchAccessor>(expressionId);
    }

    TTopologySelector::TPodSet TTopologySelector::GetPods(TExpressionId expressionId) const noexcept {
        return Impl->GetObjects<TExpressionObjects::TPodAccessor>(expressionId);
    }

    TTopologySelector::TLineSet TTopologySelector::GetLines(TExpressionId expressionId) const noexcept {
        return Impl->GetObjects<TExpressionObjects::TLineAccessor>(expressionId);
    }

    TTopologySelector::TDatacenterSet TTopologySelector::GetDatacenters(TExpressionId expressionId) const noexcept {
        return Impl->GetObjects<TExpressionObjects::TDatacenterAccessor>(expressionId);
    }

    TExpressionId TTopologySelector::DefaultExpressionId() noexcept {
        return DEFAULT_EXPRESSION_ID;
    }

    const TSet<TGroupKey>& TTopologySelector::GetGroups() const noexcept {
        return Impl->GetGroups();
    }

    TInstant TTopologySelector::Created() const noexcept {
        return Impl->Created();
    }
}
