#pragma once

#include <infra/netmon/topology/topology.h>
#include <infra/netmon/topology/clients/groups.h>
#include <infra/netmon/library/boxes.h>

#include <infra/netmon/expression/parser.h>

namespace NNetmon {
    TSet<TGroupKey> ExtractGroups(const TExpressionDnf& expr);

    class TTopologySelector : public TNonCopyable, public TAtomicRefCount<TTopologySelector> {
    public:
        using TBox = TIntrusiveLockedBox<TTopologySelector>;
        using TRef = TBox::TConstValueRef;

        using THostSet = THashSet<TTopology::THostRef>;
        using TSwitchSet = THashSet<TTopology::TSwitchRef>;
        using TPodSet = THashSet<TTopology::TPodRef>;
        using TLineSet = THashSet<TTopology::TLineRef>;
        using TDatacenterSet = THashSet<TTopology::TDatacenterRef>;

        TTopologySelector(TTopology::TBox::TConstValueRef topology,
                          const TVector<TExpressionDnf>& expressions,
                          const TGroupStorage::TExistingHosts& hostsInGroups);
        ~TTopologySelector();

        const TExpressionIdList& GetAllExpressions() const noexcept;

        const TExpressionIdList* FindSourceExpressions(const THostInterface& iface) const noexcept;
        const TExpressionIdList* FindSourceExpressions(const TTopology::THostInterfaceRef& iface) const noexcept;
        const TExpressionIdList* FindSourceExpressions(const THost& host) const noexcept;
        const TExpressionIdList* FindSourceExpressions(const TTopology::THostRef& host) const noexcept;

        const TExpressionIdList* FindTargetExpressions(const THostInterface& iface) const noexcept;
        const TExpressionIdList* FindTargetExpressions(const TTopology::THostInterfaceRef& iface) const noexcept;
        const TExpressionIdList* FindTargetExpressions(const THost& host) const noexcept;
        const TExpressionIdList* FindTargetExpressions(const TTopology::THostRef& host) const noexcept;

        ui32 GetWeight(const TSwitch& switch_, TExpressionId expressionId) const noexcept;
        ui32 GetWeight(const TTopology::TSwitchRef& switch_, TExpressionId expressionId) const noexcept;

        class IHostFilter {
        public:
            virtual ~IHostFilter() = default;

            virtual bool IsTrivial() const {
                return false;
            }

            virtual bool operator()(TTopology::THostRef) const = 0;
        };

        class TTrivialHostFilter final : public IHostFilter {
        public:
            virtual bool IsTrivial() const override {
                return true;
            }

            virtual bool operator()(TTopology::THostRef) const override {
                return true;
            }
        };

        class THostFilterByDatacenter : public IHostFilter {
        public:
            THostFilterByDatacenter(TTopology::TDatacenterRef datacenter)
                : Datacenter(datacenter)
            {
            }

            bool operator()(TTopology::THostRef host) const override {
                return host.GetDatacenter() == Datacenter;
            }

        private:
            TTopology::TDatacenterRef Datacenter;
        };

        class THostFilterByLine : public IHostFilter {
        public:
            THostFilterByLine(TTopology::TLineRef line)
                : Line(line)
            {
            }

            bool operator()(TTopology::THostRef host) const override {
                return host.GetLine() == Line;
            }

        private:
            TTopology::TLineRef Line;
        };

        class THostFilterByPod : public IHostFilter {
        public:
            THostFilterByPod(TTopology::TPodRef pod)
                : Pod(pod)
            {
            }

            bool operator()(TTopology::THostRef host) const override {
                return host.GetPod() == Pod;
            }

        private:
            TTopology::TPodRef Pod;
        };

        class THostFilterBySwitch : public IHostFilter {
        public:
            THostFilterBySwitch(TTopology::TSwitchRef switch_)
                : Switch(switch_)
            {
            }

            bool operator()(TTopology::THostRef host) const override {
                return host.GetSwitch() == Switch;
            }

        private:
            TTopology::TSwitchRef Switch;
        };

        size_t CountHosts(TExpressionId expressionId, const IHostFilter& filter=TTrivialHostFilter()) const noexcept;
        THostSet GetHosts(TExpressionId expressionId, const IHostFilter& filter=TTrivialHostFilter()) const noexcept;
        THostSet GetHosts(const TExpressionDnf& expression, const IHostFilter& filter=TTrivialHostFilter()) const noexcept;

        TSwitchSet GetSwitches(TExpressionId expressionId) const noexcept;
        TPodSet GetPods(TExpressionId expressionId) const noexcept;
        TLineSet GetLines(TExpressionId expressionId) const noexcept;
        TDatacenterSet GetDatacenters(TExpressionId expressionId) const noexcept;

        static TExpressionId DefaultExpressionId() noexcept;

        const TSet<TGroupKey>& GetGroups() const noexcept;

        TInstant Created() const noexcept;

    private:
        class TImpl;
        THolder<TImpl> Impl;
    };
}
