#include <infra/netmon/topology/topology_selector.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TTopologySelectorTest: public TTestBase {
    UNIT_TEST_SUITE(TTopologySelectorTest);
    UNIT_TEST(TestEmptySelector)
    UNIT_TEST(TestVlanSelector)
    UNIT_TEST(TestNotVlanSelector)
    UNIT_TEST(TestVirtualSelector)
    UNIT_TEST(TestVrfSelector)
    UNIT_TEST(TestGroupSelectors)
    UNIT_TEST(TestDatacenterSelector)
    UNIT_TEST(TestQueueSelector)
    UNIT_TEST(TestSwitchSelector)
    UNIT_TEST(TestWalleProjectSelector)
    UNIT_TEST(TestWalleTagSelector)
    UNIT_TEST(TestNegativeSelector)
    UNIT_TEST(TestDirection)
    UNIT_TEST_SUITE_END();

public:
    TTopologySelectorTest() {
    }

    void SetUp() override {
        Topology = TGlobalTopology::GetTopologyStorage().GetTopology();
        FirstHost = Topology.Get()->FindHost("sas1-1234.search.yandex.net");
        SecondHost = Topology.Get()->FindHost("ns8.yandex.net");
        ThirdHost = Topology.Get()->FindHost("myt1-6777.search.yandex.net");
        FirstIface = FirstHost->GetBackboneInterface();
        SecondIface = SecondHost->GetBackboneInterface();
        ThirdIface = ThirdHost->GetBackboneInterface();
    }

private:
    void Contains(const TExpressionIdList& variants, TExpressionId needle) {
        UNIT_ASSERT(Find(variants, needle) != variants.end());
    }

    void Contains(const TExpressionIdList* variants, TExpressionId needle) {
        Contains(*variants, needle);
    }

    inline void CheckHostIsMissing(TTopologySelector::TBox& selector, const THostInterface* iface) {
        UNIT_ASSERT(iface);
        const auto* sourceHostExpressions(selector.Own()->FindSourceExpressions(*iface));
        UNIT_ASSERT(!sourceHostExpressions);
        const auto* targetHostExpressions(selector.Own()->FindTargetExpressions(*iface));
        UNIT_ASSERT(!targetHostExpressions);
    }

    inline void CheckHostInDefaultExpression(const TExpressionIdList* hostExpressions, TExpressionId defaultExpressionId) {
        UNIT_ASSERT(hostExpressions);
        UNIT_ASSERT_EQUAL(hostExpressions->size(), 1);
        Contains(hostExpressions, defaultExpressionId);
    }

    inline void CheckHostInDefaultExpression(TTopologySelector::TBox& selector, const THostInterface* iface) {
        UNIT_ASSERT(iface);
        const auto defaultExpressionId(selector.Own()->DefaultExpressionId());
        CheckHostInDefaultExpression(selector.Own()->FindSourceExpressions(*iface), defaultExpressionId);
        CheckHostInDefaultExpression(selector.Own()->FindTargetExpressions(*iface), defaultExpressionId);
    }

    inline void CheckHostInGivenExpression(const TExpressionIdList* hostExpressions, TExpressionId defaultExpressionId, const TExpressionDnf& expression) {
        UNIT_ASSERT(hostExpressions);
        UNIT_ASSERT_EQUAL(hostExpressions->size(), 2);
        Contains(hostExpressions, defaultExpressionId);
        Contains(hostExpressions, expression.hash());
    }

    inline void CheckHostInGivenExpression(TTopologySelector::TBox& selector, const THostInterface* iface, const TExpressionDnf& expression) {
        const auto defaultExpressionId(selector.Own()->DefaultExpressionId());
        CheckHostInGivenExpression(selector.Own()->FindSourceExpressions(*iface), defaultExpressionId, expression);
        CheckHostInGivenExpression(selector.Own()->FindTargetExpressions(*iface), defaultExpressionId, expression);
    }

    inline void CheckForOnlyOneExpression(const TExpressionIdList* hostExpressions, TExpressionId expressionId) {
        UNIT_ASSERT(hostExpressions);
        UNIT_ASSERT_EQUAL(hostExpressions->size(), 1);
        Contains(hostExpressions, expressionId);
    }

    inline void CheckForOnlyOneExpression(const TExpressionIdList* hostExpressions, const TExpressionDnf& expression) {
        CheckForOnlyOneExpression(hostExpressions, expression.hash());
    }

    inline void TestEmptySelector() {
        TTopologySelector::TBox selector(Topology, EmptyExpressions, EmptyGroups);
        CheckHostInDefaultExpression(selector, FirstIface);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestVlanSelector() {
        TExpressionDnf expression(ParseExpression("vlan=604"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestNotVlanSelector() {
        TExpressionDnf expression(ParseExpression("not vlan=604"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInDefaultExpression(selector, FirstIface);
        CheckForOnlyOneExpression(selector.Own()->FindSourceExpressions(*SecondIface), expression);
        CheckForOnlyOneExpression(selector.Own()->FindTargetExpressions(*SecondIface), expression);
    }

    inline void TestVirtualSelector() {
        TExpressionDnf expression(ParseExpression("virtual=true"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInDefaultExpression(selector, FirstIface);
        CheckForOnlyOneExpression(selector.Own()->FindSourceExpressions(*SecondIface), expression);
        CheckForOnlyOneExpression(selector.Own()->FindTargetExpressions(*SecondIface), expression);
    }

    inline void TestVrfSelector() {
        TExpressionDnf expression(ParseExpression("vrf=Hbf"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void CheckGroupSelector(const TString& textExpression, const TGroupKey& groupKey) {
        TExpressionDnf expression(ParseExpression(textExpression));
        TGroupStorage::THostsBox hosts(TGroupStorage::THosts{FirstHost->GetName()});
        TGroupStorage::TExistingHosts groups;
        groups.emplace_back(groupKey, hosts.Get());
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            groups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestGroupSelectors() {
        CheckGroupSelector("group=K@group_name", TGroupKey::Skynet("K@group_name"));
        CheckGroupSelector("nanny=some_service", TGroupKey::Nanny("some_service"));
        CheckGroupSelector("gencfg=ALL_SEARCH", TGroupKey::Gencfg("ALL_SEARCH"));
    }

    inline void TestDatacenterSelector() {
        TExpressionDnf expression(ParseExpression("dc=sas"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestQueueSelector() {
        TExpressionDnf expression(ParseExpression("queue=sas1.4"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestSwitchSelector() {
        TExpressionDnf expression(ParseExpression("switch=sas1-s930"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestWalleProjectSelector() {
        TExpressionDnf expression(ParseExpression("walle_project=rtc"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestWalleTagSelector() {
        TExpressionDnf expression(ParseExpression("walle_tag=rtc"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostIsMissing(selector, SecondIface);
    }

    inline void TestNegativeSelector() {
        TExpressionDnf expression(ParseExpression("vrf=Hbf - dc=myt"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        CheckHostInGivenExpression(selector, FirstIface, expression);
        CheckHostInDefaultExpression(selector, ThirdIface);
    }

    inline void TestDirection() {
        TExpressionDnf expression(ParseExpression("(switch=sas1-s930 and direction=source) or (switch=myt6-s2 and direction=target)"));
        TTopologySelector::TBox selector(
            Topology,
            TVector<TExpressionDnf>{expression},
            EmptyGroups
        );
        const auto defaultExpressionId(selector.Own()->DefaultExpressionId());
        CheckHostInGivenExpression(selector.Own()->FindSourceExpressions(*FirstIface), defaultExpressionId, expression);
        CheckHostInDefaultExpression(selector.Own()->FindSourceExpressions(*ThirdIface), defaultExpressionId);
        CheckHostInGivenExpression(selector.Own()->FindTargetExpressions(*ThirdIface), defaultExpressionId, expression);
        CheckHostInDefaultExpression(selector.Own()->FindTargetExpressions(*FirstIface), defaultExpressionId);
        CheckHostIsMissing(selector, SecondIface);
    }

    TTopology::TBox::TConstValueRef Topology;

    const THost* FirstHost;
    const THost* SecondHost;
    const THost* ThirdHost;

    const THostInterface* FirstIface;
    const THostInterface* SecondIface;
    const THostInterface* ThirdIface;

    const TVector<TExpressionDnf> EmptyExpressions;
    const TGroupStorage::TExistingHosts EmptyGroups;
};

UNIT_TEST_SUITE_REGISTRATION(TTopologySelectorTest);
