#include <infra/netmon/topology/topology_storage.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/topology/metrics.h>

namespace NNetmon {
    namespace {
        class TTopologyVersions : public TNonCopyable {
        public:
            inline void Add(const TTopology::TBox::TValueRef& topology) noexcept {
                Existing.emplace_back(topology);
                AtomicSet(Last, topology.Get());
            }

            inline bool Empty() const noexcept {
                return AtomicGet(Last) == nullptr;
            }

            inline const TTopology* Get() const noexcept {
                return AtomicGet(Last);
            }

            inline void Cleanup() {
                // Always keep two latest versions of topology because
                // we don't check that it's still used in find* methods
                // and believe that topologies can't be updated so fast
                size_t initialSize = Existing.size();
                if (initialSize <= 2) {
                    return;
                }

                TVector<TTopology::TRef> usedTopologies;
                for (auto it = Existing.cbegin(); it != Existing.cend(); ++it) {
                    if (it >= Existing.cend() - 2 || IsTopologyUsed(**it)) {
                        usedTopologies.emplace_back(*it);
                    }
                }
                Existing.swap(usedTopologies);
                INFO_LOG << "Topology version cleanup finished, " << Existing.size() << " versions left, " << initialSize - Existing.size() << " erased" << Endl;
            }

        private:
            inline bool IsTopologyUsed(const TTopology& topology) {
                // TODO: convert this to something more generic
                for (const auto& datacenter : topology.GetDatacenters()) {
                    if (datacenter->StillUsed()) {
                        DEBUG_LOG << "Topology version held by datacenter ref" << Endl;
                        return true;
                    }
                    for (const auto& pod : datacenter->GetPods()) {
                        if (pod->StillUsed()) {
                            DEBUG_LOG << "Topology version held by pod ref" << Endl;
                            return true;
                        }
                    }
                    for (const auto& queue_ : datacenter->GetLines()) {
                        if (queue_->StillUsed()) {
                            DEBUG_LOG << "Topology version held by queue ref" << Endl;
                            return true;
                        }
                        for (const auto& switch_ : queue_->GetSwitches()) {
                            if (switch_->StillUsed()) {
                                DEBUG_LOG << "Topology version held by switch ref" << Endl;
                                return true;
                            }
                            for (const auto& host : switch_->GetHosts()) {
                                if (host->StillUsed()) {
                                    DEBUG_LOG << "Topology version held by host ref" << Endl;
                                    return true;
                                }
                                for (const auto& iface : host->GetInterfaces()) {
                                    if (iface->StillUsed()) {
                                        DEBUG_LOG << "Topology version held by iface ref" << Endl;
                                        return true;
                                    }
                                }
                            }
                        }
                    }
                }
                return false;
            }

            TVector<TTopology::TRef> Existing;
            TTopology* Last = nullptr;
        };
    }

    class TTopologyStorage::TImpl : public TScheduledTask {
    public:
        TImpl(const TExpressionStorage& expressionStorage, const TGroupStorage& groupStorage)
            : TScheduledTask(TTopologySettings::Get()->GetTopologyRebuildInterval())
            , ExpressionStorage(expressionStorage)
            , GroupStorage(groupStorage)
            , OutdatedTopology(0)
            , OutdatedSelector(0)
            , EventHub(TVoidEventHub::Make())
        {
            Reload();
        }

        TThreadPool::TFuture Run() override {
            return THeavyTaskThread::Get()->Add([this]() {
                Reload();
            }, false);
        }

        void Reload(bool force=false) {
            TUnistatTimer reloadTimer{TUnistat::Instance(), ETopologySignals::TopologyReloadTime};

            bool changed = false;
            if (!Topology.Get() || AtomicSwap(&OutdatedTopology, 0) || force) {
                const TFile topologyFile(TTopologySettings::Get()->GetTopologyFile(), EOpenModeFlag::OpenExisting | EOpenModeFlag::RdOnly | EOpenModeFlag::Seq);
                auto newTopology(MakeIntrusive<TTopology>(ExistedNames));
                newTopology->ParseFile(topologyFile, ShouldStop());
                Topology.Swap(newTopology);
                TopologyVersions.Add(Topology.Get());
                changed = true;
            }

            if (!TopologySelector.Get() || AtomicSwap(&OutdatedSelector, 0) || changed) {
                TUnistatTimer indexTimer{TUnistat::Instance(), ETopologySignals::TopologyIndexGenerationTime};
                auto topologySelector(MakeIntrusive<TTopologySelector>(
                    Topology.Get(),
                    ExpressionStorage.GetExpressionDnfs(),
                    GroupStorage.ExistingGroups()
                ));
                TopologySelector.Swap(topologySelector);
                changed = true;
            }

            const auto topologySelector(TopologySelector.Get());
            for (const auto& groupKey : topologySelector->GetGroups()) {
                GroupStorage.FetchGroup(groupKey);
            }

            if (changed) {
                EventHub->Notify();
            }

            CleanupTopologies();
        }

        inline const TTopology* GetTopology() const noexcept {
            return TopologyVersions.Get();
        }
        inline TTopology::TRef GetTopologyRef() const noexcept {
            return Topology.Get();
        }
        inline TTopologySelector::TRef GetTopologySelectorRef() const noexcept {
            return TopologySelector.Get();
        }

        inline bool IsValidName(const TString& name) const noexcept {
            return ExistedNames.Exists(name);
        }

        bool IsExpressionResolved(TExpressionId expressionId) const noexcept {
            const auto expressionDnf(ExpressionStorage.GetExpressionDnf(expressionId));
            if (!expressionDnf.Defined()) {
                return false;
            }

            for (const auto& groupKey : ExtractGroups(*expressionDnf)) {
                const auto state(GroupStorage.FetchGroup(groupKey));
                if (!state.Ready || !state.Error.empty()) {
                    return false;
                }
            }

            return true;
        }

        inline void UpdateTopology() noexcept {
            AtomicSet(OutdatedTopology, 1);
            Spin();
        }

        inline void UpdateSelector() noexcept {
            AtomicSet(OutdatedSelector, 1);
            Spin();
        }

        const TVoidEventHub& OnChanged() const noexcept {
            return *EventHub;
        }

    private:
        inline void CleanupTopologies() {
            TUnistatTimer cleanupTimer{TUnistat::Instance(), ETopologySignals::TopologyCleanupTime};
            TopologyVersions.Cleanup();
        }

        const TExpressionStorage& ExpressionStorage;
        const TGroupStorage& GroupStorage;

        mutable TExistedNames ExistedNames;

        TTopology::TBox Topology;
        TTopologyVersions TopologyVersions;
        TTopologySelector::TBox TopologySelector;

        TAtomic OutdatedTopology;
        TAtomic OutdatedSelector;

        TVoidEventHub::TRef EventHub;
    };

    TTopologyStorage::TTopologyStorage(const TTopologyUpdater& topologyUpdater,
                                       const TExpressionStorage& expressionStorage,
                                       const TGroupStorage& groupStorage,
                                       bool schedule)
        : Impl(MakeHolder<TImpl>(expressionStorage, groupStorage))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
        , TopologyResourceSubscription(topologyUpdater.OnChanged().Subscribe([this]() {
            DEBUG_LOG << "Topology resource changed, reloading..." << Endl;
            Impl->UpdateTopology();
        }))
        , ExpressionSubscription(expressionStorage.OnChanged().Subscribe([this]() {
            DEBUG_LOG << "Expressions changed, reloading..." << Endl;
            Impl->UpdateSelector();
        }))
        , GroupSubscription(groupStorage.OnChanged().Subscribe([this]() {
            DEBUG_LOG << "Groups changed, reloading..." << Endl;
            Impl->UpdateSelector();
        }))
    {
    }

    TTopologyStorage::~TTopologyStorage() {
    }

    TTopology::THostInterfaceRef TTopologyStorage::FindHostInterface(const TString& iface) const noexcept {
        return TTopology::THostInterfaceRef(Impl->GetTopology()->FindHostInterface(iface));
    }

    TTopology::THostInterfaceRef TTopologyStorage::FindHostInterface(const NCommon::THostInterface& iface) const noexcept {
        return TTopology::THostInterfaceRef(Impl->GetTopology()->FindHostInterface(iface));
    }

    TTopology::THostRef TTopologyStorage::FindHost(const TString& host) const noexcept {
        return TTopology::THostRef(Impl->GetTopology()->FindHost(host));
    }

    TTopology::THostRef TTopologyStorage::FindHost(const NCommon::THost& host) const noexcept {
        return TTopology::THostRef(Impl->GetTopology()->FindHost(host));
    }

    TTopology::TDatacenterRef TTopologyStorage::FindDatacenter(const TString& datacenterName) const noexcept {
        return TTopology::TDatacenterRef(Impl->GetTopology()->FindDatacenter(datacenterName));
    }

    TTopology::TDatacenterRef TTopologyStorage::FindDatacenter(const NCommon::TDatacenter& datacenter) const noexcept {
        return TTopology::TDatacenterRef(Impl->GetTopology()->FindDatacenter(datacenter));
    }

    TTopology::TLineRef TTopologyStorage::FindQueue(const TString& datacenterName,
                                                     const TString& queueName) const noexcept {
        auto topology(Impl->GetTopology());
        const auto* datacenter = topology->FindDatacenter(datacenterName);
        if (datacenter != nullptr) {
            return TTopology::TLineRef(datacenter->FindQueue(queueName));
        }
        return TTopology::TLineRef();
    }

    TTopology::TLineRef TTopologyStorage::FindQueue(const NCommon::TLine& queue_) const noexcept {
        return TTopology::TLineRef(Impl->GetTopology()->FindQueue(queue_));
    }

    TTopology::TLineRef TTopologyStorage::FindQueue(const TString& queueName) const noexcept {
        return TTopology::TLineRef(Impl->GetTopology()->FindQueue(queueName));
    }

    TTopology::TPodRef TTopologyStorage::FindPod(const TString& datacenterName,
                                                 const TString& podName) const noexcept {
        auto topology(Impl->GetTopology());
        const auto* datacenter = topology->FindDatacenter(datacenterName);
        if (datacenter != nullptr) {
            return TTopology::TPodRef(datacenter->FindPod(podName));
        }
        return TTopology::TPodRef();
    }

    TTopology::TPodRef TTopologyStorage::FindPod(const TString& podName) const noexcept {
        return TTopology::TPodRef(Impl->GetTopology()->FindPod(podName));
    }

    TTopology::TSwitchRef TTopologyStorage::FindSwitch(const TString& datacenterName,
                                                       const TString& queueName,
                                                       const TString& switchName) const noexcept {
        auto topology(Impl->GetTopology());
        const auto* datacenter = topology->FindDatacenter(datacenterName);
        if (datacenter != nullptr) {
            const auto* queue_ = datacenter->FindQueue(queueName);
            if (queue_ != nullptr) {
                return TTopology::TSwitchRef(queue_->FindSwitch(switchName));
            }
        }
        return TTopology::TSwitchRef();
    }

    TTopology::TSwitchRef TTopologyStorage::FindSwitch(const NCommon::TSwitch& switch_) const noexcept {
        return TTopology::TSwitchRef(Impl->GetTopology()->FindSwitch(switch_));
    }

    TTopology::TSwitchRef TTopologyStorage::FindSwitch(const TString& switchName) const noexcept {
        return TTopology::TSwitchRef(Impl->GetTopology()->FindSwitch(switchName));
    }

    TTopology::TBox::TConstValueRef TTopologyStorage::GetTopology() const noexcept {
        return Impl->GetTopologyRef();
    }

    TTopologySelector::TBox::TConstValueRef TTopologyStorage::GetTopologySelector() const noexcept {
        return Impl->GetTopologySelectorRef();
    }

    TExpressionId TTopologyStorage::DefaultExpressionId() const noexcept {
        return Impl->GetTopologySelectorRef()->DefaultExpressionId();
    }

    bool TTopologyStorage::IsValidName(const TString& name) const noexcept {
        return Impl->IsValidName(name);
    }

    bool TTopologyStorage::IsExpressionResolved(TExpressionId expressionId) const noexcept {
        return Impl->IsExpressionResolved(expressionId);
    }

    void TTopologyStorage::ReloadTopology() {
        return Impl->Reload(true);
    }

    const TVoidEventHub& TTopologyStorage::OnChanged() const noexcept {
        return Impl->OnChanged();
    }
}
