#pragma once

#include <infra/netmon/topology/topology.h>
#include <infra/netmon/topology/topology_selector.h>
#include <infra/netmon/topology/topology_updater.h>
#include <infra/netmon/topology/expression_storage.h>
#include <infra/netmon/topology/clients/groups.h>
#include <infra/netmon/library/event_hub.h>

namespace NNetmon {
    class TTopologyStorage : public TNonCopyable {
    public:
        using THostSet = THashSet<TTopology::THostRef>;
        using THostSetBox = TAtomicLockedBox<THostSet>;
        using THostSetRef = THostSetBox::TConstValueRef;

        using THostIdSet = THashSet<ui64>;
        using THostIdSetBox = TAtomicLockedBox<THostIdSet>;
        using THostIdSetRef = THostIdSetBox::TConstValueRef;

        TTopologyStorage(const TTopologyUpdater& topologyUpdater,
                         const TExpressionStorage& expressionStorage,
                         const TGroupStorage& groupStorage,
                         bool schedule=true);
        ~TTopologyStorage();

        TTopology::THostInterfaceRef FindHostInterface(const TString& iface) const noexcept;
        TTopology::THostInterfaceRef FindHostInterface(const NCommon::THostInterface& iface) const noexcept;

        TTopology::THostRef FindHost(const TString& host) const noexcept;
        TTopology::THostRef FindHost(const NCommon::THost& host) const noexcept;

        TTopology::TDatacenterRef FindDatacenter(const TString& datacenterName) const noexcept;
        TTopology::TDatacenterRef FindDatacenter(const NCommon::TDatacenter& datacenter) const noexcept;

        TTopology::TLineRef FindQueue(const TString& datacenterName, const TString& queueName) const noexcept;
        TTopology::TLineRef FindQueue(const NCommon::TLine& queue_) const noexcept;
        TTopology::TLineRef FindQueue(const TString& queueName) const noexcept;

        TTopology::TPodRef FindPod(const TString& datacenterName, const TString& podName) const noexcept;
        TTopology::TPodRef FindPod(const TString& podName) const noexcept;

        TTopology::TSwitchRef FindSwitch(const TString& datacenterName,
                                         const TString& queueName,
                                         const TString& switchName) const noexcept;
        TTopology::TSwitchRef FindSwitch(const NCommon::TSwitch& switch_) const noexcept;
        TTopology::TSwitchRef FindSwitch(const TString& switchName) const noexcept;

        TTopology::TRef GetTopology() const noexcept;
        TTopologySelector::TRef GetTopologySelector() const noexcept;

        TExpressionId DefaultExpressionId() const noexcept;
        bool IsValidName(const TString& name) const noexcept;
        bool IsExpressionResolved(TExpressionId expressionId) const noexcept;

        void ReloadTopology();

        const TVoidEventHub& OnChanged() const noexcept;

    private:
        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;

        TVoidEventHub::TSubscriptionGuard TopologyResourceSubscription;
        TVoidEventHub::TSubscriptionGuard ExpressionSubscription;
        TVoidEventHub::TSubscriptionGuard GroupSubscription;
    };
}
