#include <infra/netmon/topology/topology_updater.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/fences.h>
#include <infra/netmon/library/futures.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/zookeeper.h>

#include <library/cpp/neh/http_common.h>
#include <library/cpp/json/json_reader.h>

#include <util/system/file.h>
#include <util/stream/file.h>
#include <util/folder/path.h>
#include <util/generic/guid.h>

namespace NNetmon {
    namespace NImpl {
        NThreading::TFuture<TTopologyUpdater::TInfo> FetchInfoFromSandbox() {
            NHttp::TFetchOptions options;
            if (TTopologySettings::Get()->GetSandboxToken()) {
                options.SetOAuthToken(TTopologySettings::Get()->GetSandboxToken());
            }
            return THttpRequester::Get()->MakeRequest(
                TTopologySettings::Get()->GetSandboxTopologyUrl(),
                {},
                options
            )
                .Apply([](const THttpRequester::TFuture& future) {
                    NJson::TJsonValue root;
                    NJson::ReadJsonFastTree(future.GetValue()->Data, &root, true);
                    const auto& item(root["items"].GetArraySafe().at(0));
                    return TTopologyUpdater::TInfo(
                        item["skynet_id"].GetStringSafe(),
                        item["attributes"]["mds_http"].GetStringSafe(),
                        item["attributes"]["mds_https"].GetStringSafe()
                    );
                });
        }

        struct TDownloadInfo {
            TString LastModified;
        };

        NThreading::TFuture<TDownloadInfo> DownloadTopology(const TString& url, const TString& path, const TString& lastModified = "") {
            TVector<TString> headers;
            if (!lastModified.empty()) {
                headers.push_back("If-Modified-Since: " + lastModified);
            }
            return THttpRequester::Get()->MakeRequest(url, headers)
                .Apply([path](const THttpRequester::TFuture& future) {
                    auto result = future.GetValue();

                    const TFsPath dest(path);
                    const TFsPath temporary(dest.Parent() / CreateGuidAsString());
                    const TFile topologyFile(temporary.GetPath(), EOpenModeFlag::CreateAlways | EOpenModeFlag::WrOnly);
                    TUnbufferedFileOutput stream(topologyFile);
                    stream.Write(result->Data);
                    stream.Flush();
                    stream.Finish();
                    temporary.ForceRenameTo(dest.GetPath());

                    auto* lastModified = result->Headers.FindHeader("Last-Modified");
                    if (lastModified) {
                        return TDownloadInfo{lastModified->Value()};
                    } else {
                        return TDownloadInfo{};
                    }
                });
        }
    }

    TTopologyUpdater::TInfo::TInfo(
            const TString& skynetId,
            const TString& plainMDSLink,
            const TString& secureMDSLink)
        : SkynetId(skynetId)
        , PlainMDSLink(plainMDSLink)
        , SecureMDSLink(secureMDSLink)
    {
    }

    TTopologyUpdater::TInfo::TInfo(const TInfo& info)
        : SkynetId(info.SkynetId)
        , PlainMDSLink(info.PlainMDSLink)
        , SecureMDSLink(info.SecureMDSLink)
    {
    }

    TTopologyUpdater::TInfo::TInfo(const NTopology::TInfo& info)
        : SkynetId(info.SkynetId()->data(), info.SkynetId()->size())
        , PlainMDSLink(info.PlainMDSLink()->data(), info.PlainMDSLink()->size())
        , SecureMDSLink(info.SecureMDSLink()->data(), info.SecureMDSLink()->size())
    {
    }

    flatbuffers::Offset<NTopology::TInfo> TTopologyUpdater::TInfo::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NTopology::CreateTInfo(
            builder,
            builder.CreateString(SkynetId.data(), SkynetId.size()),
            builder.CreateString(PlainMDSLink.data(), PlainMDSLink.size()),
            builder.CreateString(SecureMDSLink.data(), SecureMDSLink.size())
        );
    }

    void TTopologyUpdater::TInfo::ToProto(NClient::TTopologyInfo& info) const {
        info.SetSkynetId(SkynetId);
        info.SetPlainMDSLink(PlainMDSLink);
        info.SetSecureMDSLink(SecureMDSLink);
    }

    class TTopologyUpdater::IImpl : public TScheduledTask, public TAtomicRefCount<TTopologyUpdater::IImpl> {
    public:
        using TRef = TIntrusivePtr<IImpl>;

        using TScheduledTask::TScheduledTask;

        virtual TInfo::TBox::TConstValueRef GetInfo() const = 0;
        virtual const TVoidEventHub& OnChanged() const = 0;
        virtual void DownloadTopologyIfNeeded() = 0;
        virtual ~IImpl() = default;
    };

    class TSandboxTopologyUpdater : public TTopologyUpdater::IImpl, public NAsyncZookeeper::IWatcher {
    public:
        using TInfo = TTopologyUpdater::TInfo;

        inline TSandboxTopologyUpdater(const TString& path)
            : TTopologyUpdater::IImpl(TTopologySettings::Get()->GetTopologyUpdateInterval())
            , ZookeeperClient(*NAsyncZookeeper::TClient::Get())
            , Path(path)
            , Prefix("/topology")
            , WatcherGuard(ZookeeperClient.CreateWatcher(Prefix, this))
            , EventHub(TVoidEventHub::Make())
        {
        }

        TThreadPool::TFuture Run() override {
            return NImpl::FetchInfoFromSandbox().Apply([this](const NThreading::TFuture<TInfo>& future) {
                auto info(SetTopologyInfo(MakeAtomicShared<TInfo>(future.GetValue())));
                INFO_LOG << "Current topology version in Sandbox: " << info->SkynetId << Endl;

                TString body;
                flatbuffers::FlatBufferBuilder builder;
                builder.Finish(info->ToProto(builder));
                WriteFlatBuffer(body, builder);

                Fence.Retain(ZookeeperClient.SetData(Prefix, body).Apply(
                    [this, body](const NThreading::TFuture<NAsyncZookeeper::TStatResult>& future) {
                        if (future.GetValue().ResultCode() == NAsyncZookeeper::NODE_NOT_EXISTS) {
                            Fence.Retain(ZookeeperClient.Create(Prefix, body).Apply(
                                [this](const NThreading::TFuture<NAsyncZookeeper::TPathResult>&) {
                                    // create watcher again in case of errors
                                    GetTopologyInfo();
                                }
                            ));
                        }
                    }
                ));
            });
        }

        void DownloadTopologyIfNeeded() override {
            if (!TFsPath(Path).Exists()) {
                FutureChain(
                    NImpl::FetchInfoFromSandbox(),
                    [this](const TInfo& info) { return NImpl::DownloadTopology(info.PlainMDSLink, Path); }
                ).GetValue(TDuration::Max());
            }
        }

        void OnEvent(const NAsyncZookeeper::TEvent& event) override {
            if (event.Path == Prefix && event.EventType == NAsyncZookeeper::EEventType::ET_NODE_DATA_CHANGED) {
                // watchers are one time trigger
                GetTopologyInfo();
            }
        }

        void OnExpired() override {
            GetTopologyInfo();
        }

        inline void GetTopologyInfo() {
            TRef ref(this);
            Fence.Retain(ZookeeperClient.GetData(Prefix, true).Subscribe(
                [ref, this](const NAsyncZookeeper::TDataResult::TFuture& future) {
                    // capture `this` to avoid dynamic casting from TRef aka TIntrusivePtr<IImpl>
                    OnTopologyInfo(future.GetValue());
                }
            ));
        }

        void OnTopologyInfo(const NAsyncZookeeper::TDataResult& result) noexcept {
            if (result.Empty()) {
                ERROR_LOG << "Can't get topology info from '" << Prefix << "', " << result.ResultCode() << Endl;
            } else {
                TFlatObject<NTopology::TInfo> info;
                ReadFlatBuffer(result.Result().Data(), info);
                if (info.Verify()) {
                    SetTopologyInfo(MakeAtomicShared<TInfo>(*info));
                }
            }
        }

        inline TInfo::TBox::TConstValueRef SetTopologyInfo(TInfo::TBox::TValueRef info) {
            bool changed = false;
            auto newInfo(TopologyInfo.Apply([&] (const TInfo::TBox::TConstValueRef& oldInfo) {
                changed = oldInfo->SkynetId != info->SkynetId;
                return info;
            }));
            if (changed) {
                INFO_LOG << "New topology found: " << newInfo->SkynetId << Endl;
                Fence.Retain(NImpl::DownloadTopology(newInfo->PlainMDSLink, Path).Subscribe([this](const NThreading::TFuture<NImpl::TDownloadInfo>& future) {
                    try {
                        future.GetValue();
                    } catch(...) {
                        ERROR_LOG << "Can't download new topology: " << CurrentExceptionMessage() << Endl;
                        return;
                    }
                    EventHub->Notify();
                }));
            }
            return newInfo;
        }

        TInfo::TBox::TConstValueRef GetInfo() const override {
            const auto info(TopologyInfo.Get());
            if (info->SkynetId.empty()) {
                ythrow yexception() << "topology info isn't ready yet";
            }
            return info;
        }

        const TVoidEventHub& OnChanged() const override {
            return *EventHub;
        }

    private:
        NAsyncZookeeper::TClient& ZookeeperClient;
        const TString Path;
        const TString Prefix;
        TInfo::TBox TopologyInfo;
        TFutureFence Fence;
        NAsyncZookeeper::TClient::TWatcherGuard WatcherGuard;
        TVoidEventHub::TRef EventHub;
    };

    class TCloudTopologyUpdater : public TTopologyUpdater::IImpl {
    public:
        using TInfo = TTopologyUpdater::TInfo;

        TCloudTopologyUpdater(const TString& path)
            : TTopologyUpdater::IImpl(
                TTopologySettings::Get()->GetTopologyUpdateInterval(),
                false,
                TTopologySettings::Get()->GetTopologyUpdateInterval()
            )
            , Path(path)
            , EventHub(TVoidEventHub::Make())
        {
        }

        TThreadPool::TFuture Run() override {
            return NImpl::DownloadTopology(
                TTopologySettings::Get()->GetS3TopologyUrl(),
                Path,
                *LastModified.Own()
            ).Apply([this](const NThreading::TFuture<NImpl::TDownloadInfo>& future) {
                auto lastModified = future.GetValue().LastModified;
                *LastModified.Own() = lastModified;

                auto info = MakeAtomicShared<TInfo>(
                    lastModified,
                    TTopologySettings::Get()->GetS3TopologyUrl(),
                    TTopologySettings::Get()->GetS3TopologyUrl()
                );
                TopologyInfo.Swap(info);

                EventHub->Notify();

                INFO_LOG << "Updated topology from S3: Last-Modified " << lastModified << Endl;
            });
        }

        TInfo::TBox::TConstValueRef GetInfo() const override {
            const auto info(TopologyInfo.Get());
            if (info->SkynetId.empty()) {
                ythrow yexception() << "topology info isn't ready yet";
            }
            return info;
        }

        const TVoidEventHub& OnChanged() const override {
            return *EventHub;
        }

        void DownloadTopologyIfNeeded() override {
            if (!TFsPath(Path).Exists()) {
                NImpl::DownloadTopology(
                    TTopologySettings::Get()->GetS3TopologyUrl(), Path
                ).GetValue(TDuration::Max());
            }
        }

    private:
        TString Path;
        TPlainLockedBox<TString> LastModified;
        TInfo::TBox TopologyInfo;
        TVoidEventHub::TRef EventHub;
    };

    TIntrusivePtr<TTopologyUpdater::IImpl> TTopologyUpdater::MakeImpl() {
        if (TTopologySettings::Get()->GetS3TopologyUrl()) {
            return MakeIntrusive<TCloudTopologyUpdater>(TTopologySettings::Get()->GetTopologyFile());
        } else {
            return MakeIntrusive<TSandboxTopologyUpdater>(TTopologySettings::Get()->GetTopologyFile());
        }
    }

    TTopologyUpdater::TTopologyUpdater(bool schedule)
        : Impl(MakeImpl())
        , SchedulerGuard(schedule && TTopologySettings::Get()->GetTopologyUpdateInterval() ? Impl->Schedule() : nullptr)
    {
        Impl->DownloadTopologyIfNeeded();
    }

    TTopologyUpdater::~TTopologyUpdater() {
    }

    TTopologyUpdater::TInfo::TBox::TConstValueRef TTopologyUpdater::GetInfo() const {
        return Impl->GetInfo();
    }

    const TVoidEventHub& TTopologyUpdater::OnChanged() const {
        return Impl->OnChanged();
    }
}
