#include <infra/netmon/topology/topology_storage.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TTopologyTest: public TTestBase {
    UNIT_TEST_SUITE(TTopologyTest);
    UNIT_TEST(TestFind)
    UNIT_TEST(TestRtcSwitch)
    UNIT_TEST(TestInterfaceGetters)
    UNIT_TEST_SUITE_END();

private:
    inline void TestFind() {
        TTopologyStorage& topologyStorage(TGlobalTopology::GetTopologyStorage());
        UNIT_ASSERT(topologyStorage.FindDatacenter("man"));
        UNIT_ASSERT(!topologyStorage.FindDatacenter("man1"));
        UNIT_ASSERT(topologyStorage.FindQueue("myt", "myt5"));
        UNIT_ASSERT(!topologyStorage.FindQueue("myt", "myt-5"));
        UNIT_ASSERT(topologyStorage.FindSwitch("myt", "myt5", "myt6-s2"));
        UNIT_ASSERT(!topologyStorage.FindSwitch("myt", "myt5", "myt6-s123"));

        TTopology::THostRef missingHost(topologyStorage.FindHost("jmon01.search.yandex.net"));
        UNIT_ASSERT(!missingHost);

        TTopology::THostRef foundHost(topologyStorage.FindHost("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(foundHost);
        UNIT_ASSERT(!foundHost->IsVirtual());
        UNIT_ASSERT(foundHost->GetName() == TStringBuf("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(foundHost->GetSwitch().GetName() == "sas1-s930");
        UNIT_ASSERT(foundHost->GetPod());
        UNIT_ASSERT(foundHost->GetPod()->GetName() == "sas-1");
        UNIT_ASSERT(foundHost->GetLine().GetName() == "sas1.4");
        UNIT_ASSERT(foundHost->GetDatacenter().GetName() == "sas");

        // old one topology should be available
        topologyStorage.ReloadTopology();
        UNIT_ASSERT(foundHost->GetName() == TStringBuf("sas1-1234.search.yandex.net"));

        // try to find switch
        TTopology::TSwitchRef foundSwitch(topologyStorage.FindSwitch("myt", "myt5", "myt5-s2"));
        UNIT_ASSERT(foundSwitch);
        UNIT_ASSERT(foundSwitch->GetName() == TStringBuf("myt5-s2"));

        TTopology::TSwitchRef missingSwitch(topologyStorage.FindSwitch("myt", "myt5", "myt3-s46"));
        UNIT_ASSERT(!missingSwitch);

        // try to find pod
        TTopology::TPodRef foundPod(topologyStorage.FindPod("sas", "sas-1"));
        UNIT_ASSERT(foundPod);
        UNIT_ASSERT(foundPod == topologyStorage.FindPod("sas-1"));
        UNIT_ASSERT(foundPod->GetName() == TStringBuf("sas-1"));
        UNIT_ASSERT(foundPod->GetSwitches().contains(TStringBuf("sas1-s930")));

        // check comparators
        TTopology::THostRef hostA(topologyStorage.FindHost("sas1-1234.search.yandex.net"));
        TTopology::THostRef hostB(topologyStorage.FindHost("jmon-test.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);
        UNIT_ASSERT(hostA != hostB);

        // test datacenter serialization
        TTopology::TDatacenterRef foundDatacenter(topologyStorage.FindDatacenter("myt"));
        const NCommon::TDatacenter serializedDatacenter(foundDatacenter->ToProto());
        TTopology::TDatacenterRef sameDatacenter(topologyStorage.FindDatacenter(serializedDatacenter));
        UNIT_ASSERT(sameDatacenter);
        UNIT_ASSERT(foundDatacenter == sameDatacenter);

        // test queue serialization
        TTopology::TLineRef foundQueue(topologyStorage.FindQueue("myt", "myt5"));
        const NCommon::TLine serializedQueue(foundQueue->ToProto());
        TTopology::TLineRef sameQueue(topologyStorage.FindQueue(serializedQueue));
        UNIT_ASSERT(sameQueue);
        UNIT_ASSERT(foundQueue == sameQueue);

        // test switch serialization
        const NCommon::TSwitch serializedSwitch(foundSwitch->ToProto());
        TTopology::TSwitchRef sameSwitch(topologyStorage.FindSwitch(serializedSwitch));
        UNIT_ASSERT(sameSwitch);
        UNIT_ASSERT(foundSwitch == sameSwitch);

        // test host serialization
        const NCommon::THost serializedHost(foundHost->ToProto());
        TTopology::THostRef sameHost(topologyStorage.FindHost(serializedHost));
        UNIT_ASSERT(sameHost);
        UNIT_ASSERT(foundHost == sameHost);

        // let's test ifaces
        TTopology::THostInterfaceRef foundIface(topologyStorage.FindHostInterface("fb-sas1-1234.search.yandex.net"));
        UNIT_ASSERT(foundIface);
        UNIT_ASSERT(foundHost == foundIface.GetHost());
        UNIT_ASSERT(foundIface->GetIpv6().ToString().StartsWith(TStringBuf("2a02:")));

        // check that virtual hosts detected properly
        TTopology::THostRef virtualHost(topologyStorage.FindHost("noc-duty.yndx.net"));
        UNIT_ASSERT(virtualHost);
        UNIT_ASSERT(virtualHost->IsVirtual());
    }

    inline void TestRtcSwitch() {
        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();
        UNIT_ASSERT(topologyStorage.FindSwitch("vla1-1s124")->IsRtc());
        UNIT_ASSERT(!topologyStorage.FindSwitch("vla1-2s33")->IsRtc());
    }

    inline void TestInterfaceGetters() {
        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();

        {
            auto host(topologyStorage.FindHost("logstore-myt.mail.yandex.net"));
            UNIT_ASSERT(host);
            UNIT_ASSERT(host->GetIpv4Interface());
            UNIT_ASSERT(!host->GetBackboneInterface());
            UNIT_ASSERT(host->GetFastboneInterface());
        }

        {
            auto host(topologyStorage.FindHost("vla1-1234.search.yandex.net"));
            UNIT_ASSERT(host);
            UNIT_ASSERT(!host->GetIpv4Interface());
            UNIT_ASSERT(host->GetBackboneInterface());
            UNIT_ASSERT(host->GetFastboneInterface());
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(TTopologyTest);
