#include <infra/netmon/user_data_storage.h>

#include <infra/netmon/library/mongo.h>
#include <infra/netmon/library/settings.h>

#include <util/string/builder.h>

namespace NNetmon {
    class TUserDataStorage::TImpl: public TScheduledTask {
    public:
        TImpl(bool schedule)
            : TScheduledTask(TDuration::Seconds(5), true)
            , MongoClient(schedule ? NMongo::TClient::Get() : nullptr)
            , ExpressionsCollection("expressions")
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                SynchronizeExpressionData();
            });
        }

        NJson::TJsonValue GetExpressionData(const TString& id, const TVector<TString>& fields = {}) const {
            NJson::TJsonValue data(NJson::JSON_MAP);
            if (!MongoClient) {
                return data;
            }

            auto expressionData(ExpressionData.Own());
            auto currentExpressionDataIt(expressionData->find(id));
            if (currentExpressionDataIt.IsEnd()) {
                return NJson::TJsonValue(NJson::JSON_NULL);
            }
            const auto& currentExpressionData = currentExpressionDataIt->second.GetMapSafe();

            if (fields.empty()) {
                return currentExpressionDataIt->second;
            }

            for (const auto& field : fields) {
                auto valueIt = currentExpressionData.find(field);
                if (!valueIt.IsEnd()) {
                    data.InsertValue(field, valueIt->second);
                }
            }

            return data;
        }

        NThreading::TFuture<void> UpsertExpressionData(const TString& id, const NJson::TJsonValue& data) const {
            if (!MongoClient) {
                return NThreading::MakeFuture();
            }

            NJson::TJsonValue selector;
            selector["_id"] = id;
            NJson::TJsonValue updater;
            updater["$set"] = data;

            return MongoClient->Upsert(ExpressionsCollection, selector, updater).Subscribe([this, id, data](const NThreading::TFuture<void>& future) {
                if (!future.HasException()) {
                    auto expressionData = ExpressionData.Own();
                    auto& currentExpressionData = (*expressionData)[id];
                    for (const auto& it : data.GetMapSafe()) {
                        currentExpressionData[it.first] = it.second;
                    }
                }
            });
        }

        NThreading::TFuture<void> DeleteExpressionDataFields(const TString& id, const TVector<TString>& fields) const {
            if (!MongoClient) {
                return NThreading::MakeFuture();
            }

            if (fields.empty()) {
                return NThreading::MakeFuture();
            }

            NJson::TJsonValue selector;
            selector["_id"] = id;

            NJson::TJsonValue updater;
            auto& updaterArg = updater["$unset"];
            for (const auto& field : fields) {
                updaterArg.InsertValue(field, "");
            }

            return MongoClient->Update(ExpressionsCollection, selector, updater).Subscribe([this, id, fields](const NThreading::TFuture<void>& future) {
                if (!future.HasException()) {
                    auto expressionData = ExpressionData.Own();
                    auto it = expressionData->find(id);
                    if (!it.IsEnd()) {
                        for (const auto& field : fields) {
                            it->second.EraseValue(field);
                        }
                    }
                }
            });
        }

        NThreading::TFuture<void> DeleteExpressionData(const TString& id) const {
            if (!MongoClient) {
                return NThreading::MakeFuture();
            }

            NJson::TJsonValue selector;
            selector["_id"] = id;

            return MongoClient->Remove(ExpressionsCollection, selector).Subscribe([this, id](const NThreading::TFuture<void>& future) {
                if (!future.HasException()) {
                    ExpressionData.Own()->erase(id);
                }
            });
        }

        // NB: if Move and Upsert operations are performed concurrently
        // on the same expression id, data from Upsert may get lost.
        NThreading::TFuture<void> MoveExpressionData(const TString& id, const TString& newId) const {
            if (!MongoClient) {
                return NThreading::MakeFuture();
            }

            if (id == newId) {
                return NThreading::MakeFuture();
            }

            if (ExpressionData.Own()->contains(newId)) {
                auto promise(NThreading::NewPromise());
                promise.SetException(TStringBuilder() << "metadata for expression '" << newId << "' already exists");
                return promise.GetFuture();
            }

            auto data = GetExpressionData(id);
            if (data.IsNull()) {
                return NThreading::MakeFuture();
            }
            return UpsertExpressionData(newId, data).Apply([this, id](const NThreading::TFuture<void>& future) {
                if (future.HasException()) {
                    return future;
                } else {
                    return DeleteExpressionData(id);
                }
            });
        }

        bool IsQualitySignalsEnabled(const TString& id) const {
            auto expressionData(ExpressionData.Own());
            auto it(expressionData->find(id));
            if (!it.IsEnd() && it->second.Has("quality_signals_enabled")) {
                return it->second["quality_signals_enabled"].GetBoolean();
            }

            return false;
        }

    private:
        void SynchronizeExpressionData() {
            if (!MongoClient) {
                return;
            }

            MongoClient->Find(ExpressionsCollection).Subscribe([this](const NThreading::TFuture<TVector<NJson::TJsonValue>>& future) {
                try {
                    const auto documents(future.GetValue());
                    TExpressionDataMap newData;
                    for (auto document : documents) {
                        auto id(document["_id"].GetStringSafe());
                        document.EraseValue("_id");
                        newData.emplace(std::move(id), std::move(document));
                    }
                    ExpressionData.Own()->swap(newData);
                } catch (...) {
                    ERROR_LOG << "Can't synchronize expression data: " << CurrentExceptionMessage() << Endl;
                }
            });
        }

        using TExpressionDataMap = THashMap<TString, NJson::TJsonValue>;

        NMongo::TClient* MongoClient;

        const TString ExpressionsCollection;
        TPlainLockedBox<TExpressionDataMap> ExpressionData;
    };

    TUserDataStorage::TUserDataStorage()
        : TUserDataStorage(!TLibrarySettings::Get()->GetMongoUri().empty())
    {
    }

    TUserDataStorage::TUserDataStorage(bool schedule)
        : Impl(MakeHolder<TImpl>(schedule))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TUserDataStorage::~TUserDataStorage() = default;

    NJson::TJsonValue TUserDataStorage::GetExpressionData(const TString& id, const TVector<TString>& fields) const {
        return Impl->GetExpressionData(id, fields);
    }

    NThreading::TFuture<void> TUserDataStorage::UpsertExpressionData(const TString& id, const NJson::TJsonValue& data) const {
        return Impl->UpsertExpressionData(id, data);
    }

    NThreading::TFuture<void> TUserDataStorage::DeleteExpressionDataFields(const TString& id, const TVector<TString>& fields) const {
        return Impl->DeleteExpressionDataFields(id, fields);
    }

    NThreading::TFuture<void> TUserDataStorage::DeleteExpressionData(const TString& id) const {
        return Impl->DeleteExpressionData(id);
    }

    NThreading::TFuture<void> TUserDataStorage::MoveExpressionData(const TString& id, const TString& newId) const {
        return Impl->MoveExpressionData(id, newId);
    }

    bool TUserDataStorage::IsQualitySignalsEnabled(const TString& id) const {
        return Impl->IsQualitySignalsEnabled(id);
    }
}
