#include <infra/netmon/utils/common.h>
#include <infra/netmon/idl/api.fbs.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/probe.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/string/subst.h>
#include <util/system/type_name.h>

#include <contrib/libs/flatbuffers/include/flatbuffers/idl.h>

using namespace NNetmon;

namespace {
    const ui64 MICROSECONDS_IN_SECOND = 1000000UL;

    THolder<flatbuffers::Parser> LoadFlatSchema(const TString& arcadiaRoot) {
#ifdef NOC_SLA_BUILD
        TString schemaPath = arcadiaRoot + "/infra/netmon/idl/noc_sla/api.fbs";
#else
        TString schemaPath = arcadiaRoot + "/infra/netmon/idl/prod/api.fbs";
#endif
        std::string schema;
        if (!flatbuffers::LoadFile(schemaPath.c_str(), false, &schema)) {
            ythrow yexception() << "Failed to load flatbuffers schema from " << schemaPath << Endl;
        }

        flatbuffers::IDLOptions options;
        options.strict_json = true;
        THolder<flatbuffers::Parser> parser(new flatbuffers::Parser(options));
        const char* includePaths[] = { arcadiaRoot.c_str(), nullptr };
        if (!parser->Parse(schema.c_str(), includePaths)) {
            ythrow yexception() << "Failed to parse flatbuffers schema: " << parser->error_;
        }

        return parser;
    }

    NApi::ELevel ParseLevel(const TString& level) {
        if (level == "dc") {
            return NApi::ELevel_LEVEL_DATACENTER;
        } else if (level == "queue" || level == "line") {
            return NApi::ELevel_LEVEL_QUEUE;
        } else if (level == "switch") {
            return NApi::ELevel_LEVEL_SWITCH;
        } else {
            ythrow yexception() << "Unknown level " << level;
        }
    }
}

class TGatherers: public TNonCopyable {
public:
    TGatherers(flatbuffers::Parser& parser, NApi::ELevel level, TExpressionId expressionId,
               TInstant requestedTs = TInstant(), TDuration interval = TDuration())
        : Parser(parser)
        , Level(level)
        , ExpressionId(expressionId)
        , NetworkType(BACKBONE6)
        , ProtocolType(UDP)
        , AggregatorKey(ExpressionId, NetworkType, ProtocolType)
        , DatacenterPairKey(595603492658584752UL, 595603492658584752UL) // man -> man
        , LinePairKey(1587876498568487974UL, 1587876498568487974UL) // man1 -> man1
        , SwitchPairKey(16926603615096603253UL, 16926603615096603253UL) // man1-s2 -> man1-s2
        , RequestedTs(requestedTs ? requestedTs : TInstant::Seconds(TInstant::Now().Seconds() - TInstant::Seconds(3600).Seconds()))
        , Since(RequestedTs)
        , Until(interval ? requestedTs + interval : TInstant::Now())
    {
        TString path;

        // /slicer/v1/datacenter_state
        // /slicer/v1/queue_state
        // /slicer/v1/switch_state

        path = "/slicer/v1/datacenter_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            std::vector<NCommon::TAggregatorKey> keys;
            keys.push_back(AggregatorKey.ToProto());
            builder.Finish(NApi::CreateTStateRequest(builder, builder.CreateVectorOfStructs(keys)));

            return TGatherer<NApi::TDatacenterStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TDatacenterStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TDatacenterStateResponse>(future);
            });
        };

        path = "/slicer/v1/queue_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            std::vector<NCommon::TAggregatorKey> keys;
            keys.push_back(AggregatorKey.ToProto());
            builder.Finish(NApi::CreateTStateRequest(builder, builder.CreateVectorOfStructs(keys)));

            return TGatherer<NApi::TLineStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TLineStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TLineStateResponse>(future);
            });
        };

        path = "/slicer/v1/switch_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            std::vector<NCommon::TAggregatorKey> keys;
            keys.push_back(AggregatorKey.ToProto());
            builder.Finish(NApi::CreateTStateRequest(builder, builder.CreateVectorOfStructs(keys)));

            return TGatherer<NApi::TSwitchStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TSwitchStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TSwitchStateResponse>(future);
            });
        };

        // /slicer/v1/datacenter_history_state
        // /slicer/v1/queue_history_state
        // /slicer/v1/switch_history_state

        path = "/slicer/v1/datacenter_history_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTDatacenterHistoryStateRequest(builder, &AggregatorKey.ToProto(), RequestedTs.MicroSeconds()));

            return TGatherer<NApi::TDatacenterStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TDatacenterStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TDatacenterStateResponse>(future);
            });
        };

        path = "/slicer/v1/queue_history_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTLineHistoryStateRequest(builder, &AggregatorKey.ToProto(), &DatacenterPairKey, RequestedTs.MicroSeconds()));

            return TGatherer<NApi::TLineStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TLineStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TLineStateResponse>(future);
            });
        };

        path = "/slicer/v1/switch_history_state";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSwitchHistoryStateRequest(builder, &AggregatorKey.ToProto(), &LinePairKey, RequestedTs.MicroSeconds()));

            return TGatherer<NApi::TSwitchStateResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TSwitchStateResponse>::TFuture& future) {
                return PrintResponse<NApi::TSwitchStateResponse>(future);
            });
        };

        // /slicer/v1/datacenter_history_series
        // /slicer/v1/queue_history_series
        // /slicer/v1/switch_history_series

        path = "/slicer/v1/datacenter_history_series";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTDatacenterHistorySeriesRequest(builder, &AggregatorKey.ToProto(), &DatacenterPairKey, Since.MicroSeconds(), Until.MicroSeconds()));

            return TGatherer<NApi::TDatacenterHistorySeriesResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TDatacenterHistorySeriesResponse>::TFuture& future) {
                return PrintResponse<NApi::TDatacenterHistorySeriesResponse>(future);
            });
        };

        path = "/slicer/v1/queue_history_series";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTLineHistorySeriesRequest(builder, &AggregatorKey.ToProto(), &LinePairKey, Since.MicroSeconds(), Until.MicroSeconds()));

            return TGatherer<NApi::TLineHistorySeriesResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TLineHistorySeriesResponse>::TFuture& future) {
                return PrintResponse<NApi::TLineHistorySeriesResponse>(future);
            });
        };

        path = "/slicer/v1/switch_history_series";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSwitchHistorySeriesRequest(builder, &AggregatorKey.ToProto(), &SwitchPairKey, Since.MicroSeconds(), Until.MicroSeconds()));

            return TGatherer<NApi::TSwitchHistorySeriesResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TSwitchHistorySeriesResponse>::TFuture& future) {
                return PrintResponse<NApi::TSwitchHistorySeriesResponse>(future);
            });
        };

        // /slicer/v1/probes
        // /slicer/v1/append_probes

        // TODO

        // /slicer/v1/aggregator_keys
        // /slicer/v1/seen_hosts
        // /slicer/v1/terminated_hosts

        path = "/slicer/v1/aggregator_keys";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTAggregatorKeysRequest(builder, Level));

            return TGatherer<NApi::TAggregatorKeysResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TAggregatorKeysResponse>::TFuture& future) {
                return PrintResponse<NApi::TAggregatorKeysResponse>(future);
            });
        };

        path = "/slicer/v1/seen_hosts";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSeenHostsRequest(builder));

            return TGatherer<NApi::TSeenHostsResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TSeenHostsResponse>::TFuture& future) {
                return PrintResponse<NApi::TSeenHostsResponse>(future);
            });
        };

        path = "/slicer/v1/terminated_hosts";
        Gatherers[path] = [this, path]() {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTTerminatedHostsRequest(builder));

            return TGatherer<NApi::TTerminatedHostsResponse>::Collect(path, builder).Apply([this](const TGatherer<NApi::TTerminatedHostsResponse>::TFuture& future) {
                return PrintResponse<NApi::TTerminatedHostsResponse>(future);
            });
        };
    }

    ~TGatherers() = default;

    NThreading::TFuture<void> Gather(const TString& path) {
        const auto it(Gatherers.find(path));
        if (!it.IsEnd()) {
            return it->second();
        } else {
            ythrow yexception() << "Unhandled path " << path;
        }
    }

private:
    template <class T>
    NThreading::TFuture<void> PrintResponse(const typename TGatherer<T>::TFuture& future) {
        TString type(TypeName<T>());
        SubstGlobal(type, "::", ".");
        Parser.SetRootType(type.c_str());
        for (const auto& response : future.GetValue()) {
            std::string json;
            if (GenerateText(Parser, response.GetBuffer().Data(), &json)) {
                Cout << json << Endl;
            } else {
                Cerr << "Failed to parse response" << Endl;
            }
        }
        return NThreading::MakeFuture();
    }

    flatbuffers::Parser& Parser;

    NApi::ELevel Level;

    TExpressionId ExpressionId;
    ENetworkType NetworkType;
    EProtocolType ProtocolType;
    TProbeAggregatorKey AggregatorKey;
    NCommon::TDatacenterPairKey DatacenterPairKey;
    NCommon::TLinePairKey LinePairKey;
    NCommon::TSwitchPairKey SwitchPairKey;
    TInstant RequestedTs;
    TInstant Since;
    TInstant Until;

    THashMap<TString, std::function<NThreading::TFuture<void>()>> Gatherers;
};

int main(int argc, char** argv) {
    try {
        using namespace NLastGetopt;

        TOpts opts = TOpts::Default();
        opts.AddHelpOption('h');
        opts.AddVersionOption('v');

        opts.SetFreeArgsNum(1);
        opts.SetFreeArgTitle(0, "Slicer API path");

        TOpt& optConfigFile = opts.AddLongOption('c', "config", "Path to config file");
        optConfigFile.Optional().RequiredArgument().DefaultValue("netmon.json");

        TOpt& optArcadiaRoot = opts.AddLongOption('a', "arcadia", "Path to arcadia root");
        optArcadiaRoot.Optional().RequiredArgument().DefaultValue(GetEnv("ARCADIA_ROOT"));

        TOpt& optLevel = opts.AddLongOption('l', "level", "Network level: dc, queue|line, switch");
        optLevel.Optional().RequiredArgument().DefaultValue("dc");

        TOpt& optExpressionId = opts.AddLongOption('e', "expression", "Expression ID");
        optExpressionId.Optional().RequiredArgument().DefaultValue(TTopologySelector::DefaultExpressionId());

        TOpt& optTimestamp = opts.AddLongOption('t', "timestamp", "History request timestamp");
        optTimestamp.Optional().RequiredArgument().DefaultValue("0");

        TOpt& optTimeInterval = opts.AddLongOption('d', "duration", "History request duration (for history_series requests)");
        optTimeInterval.Optional().RequiredArgument().DefaultValue("0");

        TOptsParseResult res(&opts, argc, argv);

        LoadSettings(res.Get(&optConfigFile));

        auto parser(LoadFlatSchema(res.Get(&optArcadiaRoot)));
        TGatherers gatherers(*parser, ParseLevel(res.Get(&optLevel)), FromString<TExpressionId>(res.Get(&optExpressionId)),
                             TInstant::FromValue(FromString<ui64>(res.Get(&optTimestamp)) * MICROSECONDS_IN_SECOND),
                             TDuration::FromValue(FromString<ui64>(res.Get(&optTimeInterval))) * MICROSECONDS_IN_SECOND);
        gatherers.Gather(res.GetFreeArgs().front()).Wait();
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }

    return 0;
}
