#include <infra/netmon/utils/common.h>
#include <infra/netmon/idl/common.fbs.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/folder/path.h>

namespace {
    const TString SEP(",");

    inline TVector<TString> SplitNames(const TString& input) {
        TVector<TString> names;
        Split(input, SEP, names);
        return names;
    }

    inline TVector<ui64> SplitIds(const TString& input) {
        TVector<TString> ids;
        Split(input, SEP, ids);
        TVector<ui64> output;
        for (const auto& id : ids) {
            output.emplace_back(::FromString(id));
        }
        return output;
    }
}

int main(int argc, char** argv) {
    try {
        using namespace NLastGetopt;

        TString argv0 = argc ? argv[0] : "";
        auto baseDir = TFsPath(argv0).Parent();

        TOpts opts = TOpts::Default();
        opts.AddHelpOption('h');
        opts.AddVersionOption('v');

        TOpt& optConfigFile = opts.AddLongOption('c', "config", "Path to config file");
        optConfigFile.Optional().RequiredArgument().DefaultValue(baseDir / "../../netmon.json");

        NLastGetopt::TOpt& optTopologyFile = opts.AddLongOption("topology-file", "Path to topology file.");
        optTopologyFile.Optional().RequiredArgument().DefaultValue(baseDir / "topology.msgpack.gz");

        TOpt& optExpression = opts.AddLongOption('e', "expression", "Expression");
        optExpression.Optional().RequiredArgument();

        TOpt& optIfaceId = opts.AddLongOption("iface-id", "Interface ID");
        optIfaceId.Optional().RequiredArgument();
        TOpt& optIfaceName = opts.AddLongOption("iface-name", "Interface name");
        optIfaceName.Optional().RequiredArgument();

        TOpt& optHostId = opts.AddLongOption("host-id", "Host ID");
        optHostId.Optional().RequiredArgument();
        TOpt& optHostName = opts.AddLongOption("host-name", "Host name");
        optHostName.Optional().RequiredArgument();

        TOpt& optSwitchId = opts.AddLongOption("switch-id", "Switch ID");
        optSwitchId.Optional().RequiredArgument();
        TOpt& optSwitchName = opts.AddLongOption("switch-name", "Switch name");
        optSwitchName.Optional().RequiredArgument();

        TOpt& optLineId = opts.AddLongOption("line-id", "Line ID");
        optLineId.Optional().RequiredArgument();
        TOpt& optLineName = opts.AddLongOption("line-name", "Line name");
        optLineName.Optional().RequiredArgument();

        TOpt& optDatacenterId = opts.AddLongOption("datacenter-id", "Datacenter ID");
        optDatacenterId.Optional().RequiredArgument();
        TOpt& optDatacenterName = opts.AddLongOption("datacenter-name", "Datacenter name");
        optDatacenterName.Optional().RequiredArgument();

        TOptsParseResult res(&opts, argc, argv);

        LoadSettings(res.Get(&optConfigFile));
        DoInitGlobalLog("cerr", TLOG_DEBUG, false, false);

        TTopologySettings::Get()->SetTopologyFile(res.Get(opts.FindLongOption("topology-file")));
        const auto& topologyStorage(TGlobalTopology::GetTopologyStorage());
        const auto topology(topologyStorage.GetTopology());

        if (res.Has(&optExpression)) {
            const auto expression(res.Get(&optExpression));
            const auto expressionDnf(ParseExpression(expression));
            Cout << "Expression '" << expression << "': " << expressionDnf.hash() << Endl;
        }

        if (res.Has(&optIfaceId)) {
            for (const auto ifaceId : SplitIds(res.Get(&optIfaceId))) {
                NCommon::THostInterface flatIface(ifaceId);
                const auto iface(topologyStorage.FindHostInterface(flatIface));
                Cout << "Interface " << ifaceId << ": " << (iface ? iface->GetName() : "not found") << Endl;
            }
        }
        if (res.Has(&optIfaceName)) {
            for (const auto& ifaceName : SplitNames(res.Get(&optIfaceName))) {
                const auto iface(topologyStorage.FindHostInterface(ifaceName));
                Cout << "Interface " << ifaceName << ": " << (iface ? ToString(iface->GetReducedId()) : "not found") << Endl;
            }
        }

        if (res.Has(&optHostId)) {
            for (const auto hostId : SplitIds(res.Get(&optHostId))) {
                NCommon::THost flatHost(hostId);
                const auto host(topologyStorage.FindHost(flatHost));
                Cout << "Host " << hostId << ": " << (host ? host->GetName() : "not found") << Endl;
            }
        }
        if (res.Has(&optHostName)) {
            for (const auto& hostName : SplitNames(res.Get(&optHostName))) {
                const auto host(topologyStorage.FindHost(hostName));
                Cout << "Host " << hostName << ": " << (host ? ToString(host->GetReducedId()) : "not found") << Endl;
            }
        }

        if (res.Has(&optSwitchId)) {
            for (const auto switchId : SplitIds(res.Get(&optSwitchId))) {
                NCommon::TSwitch flatSwitch(switchId);
                const auto switch_(topologyStorage.FindSwitch(flatSwitch));
                Cout << "Switch " << switchId << ": " << (switch_ ? switch_->GetName() : "not found") << Endl;
            }
        }
        if (res.Has(&optSwitchName)) {
            for (const auto& switchName : SplitNames(res.Get(&optSwitchName))) {
                const auto switch_(topologyStorage.FindSwitch(switchName));
                Cout << "Switch " << switchName << ": " << (switch_ ? ToString(switch_->GetReducedId()) : "not found") << Endl;
            }
        }

        if (res.Has(&optLineId)) {
            for (const auto lineId : SplitIds(res.Get(&optLineId))) {
                NCommon::TLine flatLine(lineId);
                const auto line(topologyStorage.FindQueue(flatLine));
                Cout << "Line " << lineId << ": " << (line ? line->GetName() : "not found") << Endl;
            }
        }
        if (res.Has(&optLineName)) {
            for (const auto& lineName : SplitNames(res.Get(&optLineName))) {
                const auto line(topologyStorage.FindQueue(lineName));
                Cout << "Line " << lineName << ": " << (line ? ToString(line->GetReducedId()) : "not found") << Endl;
            }
        }

        if (res.Has(&optDatacenterId)) {
            for (const auto datacenterId : SplitIds(res.Get(&optDatacenterId))) {
                NCommon::TDatacenter flatDatacenter(datacenterId);
                const auto datacenter(topologyStorage.FindDatacenter(flatDatacenter));
                Cout << "Datacenter " << datacenterId << ": " << (datacenter ? datacenter->GetName() : "not found") << Endl;
            }
        }
        if (res.Has(&optDatacenterName)) {
            for (const auto& datacenterName : SplitNames(res.Get(&optDatacenterName))) {
                const auto datacenter(topologyStorage.FindDatacenter(datacenterName));
                Cout << "Datacenter " << datacenterName << ": " << (datacenter ? ToString(datacenter->GetReducedId()) : "not found") << Endl;
            }
        }
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }

    return 0;
}
