#!/bin/bash

source $ARCADIA_ROOT/infra/netmon/yc/error_reporting.sh

APPLY_USAGE="$0 apply deployment.yaml"
DELETE_USAGE="$0 delete deployment.yaml"
RESTART_USAGE="$0 restart deployment.yaml"
BASH_USAGE="$0 bash pod_name"
INSTALL_INGRESS="$0 install_ingress"
INSTALL_SECRET_STORE="$0 install_secret_store"

usage() {
    echo """
Usage:
    $APPLY_USAGE
    $DELETE_USAGE
    $RESTART_USAGE
    $BASH_USAGE
    $INSTALL_INGRESS
    $INSTALL_SECRET_STORE
"""
}

render_spec() {
    YAML_FILE=$1

    DNS_ZONE=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_dns_zone)
    case "$DNS_ZONE" in
    netmon.cloud.yandex.net)
        CONFIG_PATH=/yc-config-prod.json
        SOLOMON_INSTALLATION=cloud_prod
        ;;
    netmon.cloud-preprod.yandex.net)
        CONFIG_PATH=/yc-config-preprod.json
        SOLOMON_INSTALLATION=cloud_preprod
        ;;
    esac

    CR_ID=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_cr)
    YAML=$(sed "s/\<CR_ID_PLACEHOLDER\>/$CR_ID/g" $YAML_FILE)

    YAML=$(echo "$YAML" | sed "s/\<DNS_ZONE_PLACEHOLDER\>/$DNS_ZONE/g")
    YAML=$(echo "$YAML" | sed "s#\<CONFIG_PATH_PLACEHOLDER\>#$CONFIG_PATH#g")
    YAML=$(echo "$YAML" | sed "s/\<SOLOMON_INSTALLATION_PLACEHOLDER\>/$SOLOMON_INSTALLATION/g")

    if echo "$YAML" | grep -q "\<SUBNETS_PLACEHOLDER\>"; then
        SUBNETS=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_subnets | paste -sd "," -)
        YAML=$(echo "$YAML" | sed "s/\<SUBNETS_PLACEHOLDER\>/$SUBNETS/g")
    fi

    if echo "$YAML" | grep -q "\<RESERVED_ADDRESS_PLACEHOLDER\>"; then
        RESERVED_ADDRESS=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_reserved_address)
        YAML=$(echo "$YAML" | sed "s/\<RESERVED_ADDRESS_PLACEHOLDER\>/$RESERVED_ADDRESS/g")
    fi

    if echo "$YAML" | grep -q "\<WEB_CERT_PLACEHOLDER\>"; then
        WEB_CERT=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_web_cert)
        YAML=$(echo "$YAML" | sed "s/\<WEB_CERT_PLACEHOLDER\>/$WEB_CERT/g")
    fi

    if echo "$YAML" | grep -q "\<CR_ADDR_PLACEHOLDER\>"; then
        CR_ADDR=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_cr_address)
        YAML=$(echo "$YAML" | sed "s/\<CR_ADDR_PLACEHOLDER\>/$CR_ADDR/g")
    fi

    if echo "$YAML" | grep -q "\<NETMON_TOKENS_SECRET_ID\>"; then
        SECRET_ID=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_netmon_tokens_secret_id)
        YAML=$(echo "$YAML" | sed "s/\<NETMON_TOKENS_SECRET_ID\>/$SECRET_ID/g")
    fi

    echo "$YAML"
}

render_zone_spec()
{
    ZONE_YAML=$1
    ZONE=$2

    ZONE_YAML=$(echo "$ZONE_YAML" | sed "s/\<ZONE_PLACEHOLDER\>/$ZONE/g")

    if echo "$ZONE_YAML" | grep -q "\<FRONTEND_NODES_ZONE_IPS_PLACEHOLDER\>"; then
        FRONTEND_NODES_ZONE_IPS=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_frontend_nodes_zone_ips $ZONE | awk '{ print "\""$0"\""}' | paste -sd "," - | awk '{ print "["$0"]" }')
        ZONE_YAML=$(echo "$ZONE_YAML" | sed "s/\<FRONTEND_NODES_ZONE_IPS_PLACEHOLDER\>/$FRONTEND_NODES_ZONE_IPS/g")
    fi

    echo "$ZONE_YAML"
}

cmd_apply() {
    if [[ $# -ne 1 ]]; then
        echo "Usage: $APPLY_USAGE"
        exit 1
    fi

    YAML=$(render_spec $1)
    if echo "$YAML" | grep -q "\<ZONE_PLACEHOLDER\>"; then
        ZONES=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_zones)
        for zone in $ZONES
        do
            ZONE_YAML=$(render_zone_spec "$YAML" $zone)
            echo "$ZONE_YAML" | kubectl apply -f -
        done
    else
        echo "$YAML" | kubectl apply -f -
    fi
}

cmd_delete() {
    if [[ $# -ne 1 ]]; then
        echo "Usage: $DELETE_USAGE"
        exit 1
    fi

    YAML=$(render_spec $1)
    ZONES=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_zones)
    for zone in $ZONES
    do
        ZONE_YAML=$(render_zone_spec "$YAML" $zone)
        (echo "$ZONE_YAML" | kubectl delete -f -) || true
    done
}

cmd_restart() {
    if [[ $# -ne 1 ]]; then
        echo "Usage: $RESTART_USAGE"
        exit 1
    fi

    SVC=$(basename -s .yaml $1)
    SVC_TYPE=statefulset
    ZONES=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_zones)
    for zone in $ZONES
    do
        kubectl rollout restart $SVC_TYPE $SVC-$zone
    done
}

cmd_bash() {
    if [[ $# -ne 1 ]]; then
        echo "Usage: $BASH_USAGE"
        exit 1
    fi

    POD=$1
    kubectl exec --stdin --tty $POD -- /bin/bash
}

cmd_install_ingress() {
    NAMESPACE=yc-alb-ingress
    VERSION=v0.1.0
    FOLDER_ID=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_folder)
    CLUSTER_ID=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_k8s)
    ENDPOINT=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_active_sa_endpoint)

    SA_KEY_FILE=yc-sa-key.json
    $ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_active_sa_key >> $SA_KEY_FILE

    kubectl create namespace $NAMESPACE

    export HELM_EXPERIMENTAL_OCI=1
    helm pull --version $VERSION oci://cr.yandex/yc/yc-alb-ingress-controller-chart
    helm install --namespace $NAMESPACE --set folderId=$FOLDER_ID --set clusterId=$CLUSTER_ID --set-file saKeySecretKey=$SA_KEY_FILE yc-alb-ingress-controller ./yc-alb-ingress-controller-chart-$VERSION.tgz --set endpoint=$ENDPOINT

    rm -f $SA_KEY_FILE
}

cmd_install_secret_store() {
    if ! helm status --namespace external-secrets external-secrets &>/dev/null; then
        export HELM_EXPERIMENTAL_OCI=1
        VERSION=0.3.8-2
        helm pull --version $VERSION oci://cr.yandex/yc-marketplace/yandex-cloud/external-secrets/chart/external-secrets
        helm install external-secrets ./external-secrets-$VERSION.tgz --namespace external-secrets --create-namespace
    else
        echo >&2 "Skipping external-secrets installation: already installed."
    fi

    if ! kubectl get secret secret-store-auth &>/dev/null; then
        ESO_ACCOUNT=$($ARCADIA_ROOT/infra/netmon/yc/manage_cloud.sh get_external_secrets_sa)
        ESO_ACCOUNT_KEY=$ESO_ACCOUNT.json
        if [[ ! -f $ESO_ACCOUNT_KEY ]]; then
            echo >&2 "$ESO_ACCOUNT_KEY not found. Get it from https://yav.yandex-team.ru/secret/sec-01g6jftdwqdj0kvt5qganvg38v."
            exit 1
        fi

	    kubectl create secret generic secret-store-auth --from-file=authorized-key=$ESO_ACCOUNT_KEY
    fi

    kubectl apply -f - <<< '
apiVersion: external-secrets.io/v1alpha1
kind: SecretStore
metadata:
  name: secret-store
spec:
  provider:
    yandexlockbox:
      auth:
        authorizedKeySecretRef:
          name: secret-store-auth
          key: authorized-key'

    cmd_apply secrets.yaml
}

case "${1:-}" in
    "") usage;;
    apply) shift; cmd_apply "$@";;
    delete) shift; cmd_delete "$@";;
    restart) shift; cmd_restart "$@";;
    bash) shift; cmd_bash "$@";;
    install_ingress) shift; cmd_install_ingress "$@";;
    install_secret_store) shift; cmd_install_secret_store "$@";;
esac
