#!/bin/bash

source $ARCADIA_ROOT/infra/netmon/yc/error_reporting.sh

ZONES="ru-central1-a ru-central1-b ru-central1-c"
#ZONES="ru-central1-a"

DNS_TTL=60

NETMON_NETWORK_PREFIX="cloud-netmon"

NETMON_FRONTEND="netmon-frontend"
NETMON_FRONTEND_CORES=2
NETMON_FRONTEND_MEMORY="8G"
NETMON_FRONTEND_DISK="30G"
NETMON_FRONTEND_LABELS="end=front"

NETMON_BACKEND="netmon-backend"
NETMON_BACKEND_CORES=8
NETMON_BACKEND_MEMORY="32G"
NETMON_BACKEND_DISK="30G"
NETMON_BACKEND_LABELS="end=back"

#
# Zones
#
function get_zones() # Get available zones
{
    echo $ZONES
}

#
# Cloud
#

function get_cloud() # Get cloud
{
    echo "yc.netmon.service-cloud"
}

#
# Folder
#

function get_folder() # Get folder
{
    echo "yc.netmon.main-folder"
}

#
# Service account
#

function get_active_sa() # Get active service account
{
    yc config profile list | grep ACTIVE | cut -d' ' -f1
}

function get_active_sa_endpoint() # Get active service account endpoint
{
    yc config get endpoint
}

function get_active_sa_key() # Get active service account key
{
    sa_name=$(get_active_sa)
    yc config profile get $sa_name | yq -r '.["service-account-key"]'
}

function get_external_secrets_sa() # Get name of account used by external-secrets
{
    echo "external-secrets-sa"
}

#
# Certificates
#

function get_ya_internal_root_ca() # Get YandexInternalRootCA
{
    curl https://crls.yandex.net/YandexInternalRootCA.crt
}

function create_web_cert() # Create web certificate
{
    dns_zone=$(get_dns_zone)
    folder_id=$(get_folder)
    cert_req="""
description: 'Netmon Web SSL Certificate'
domains: [
    '$dns_zone',
    '*.$dns_zone'
]
folder_id: $folder_id
provider: INTERNAL_CA
"""

    ycp certificatemanager v1 certificate request-new -r - <<REQ
$cert_req
REQ
}

function get_web_cert() # Get web certificate
{
    yc certificate-manager certificate list --format json | jq -r '.[0].id'
}

function download_web_cert() # Download web certificate with private key
{
    export YC_TOKEN=$(yc iam create-token)
    kms_key=$(get_kms_key)
    cert_id=$(get_web_cert)
    endpoint=$(get_active_sa_endpoint)
    uid=$(id -u)
    gid=$(id -g)
    skm_yaml="""
kek:
  kms:
    keyUri: yc-kms://$kms_key
secrets:
- path: cert.pem
  mode: 0600
  uid: $uid
  gid: $gid
  source:
    certificateManager:
      certificateId: $cert_id
      key: certificate_and_key
yc:
  apiEndpoint: $endpoint
"""
    echo "$skm_yaml" > skm.yaml
    encrypted_dek=$(skm generate-dek --config skm.yaml)
    echo "encrypted_dek: $encrypted_dek" >> skm.yaml
    skm encrypt-bundle --config skm.yaml --bundle bundle.yaml
    skm decrypt-bundle --config skm.yaml --bundle bundle.yaml --no-chown
    rm -f bundle.yaml skm.yaml
}

#
# KMS
#

function create_kms_key() # Create KMS symmetric key
{
    yc kms symmetric-key create --name "kms_key" --default-algorithm "aes-256"
}

function get_kms_key() # Get KMS symmetric key
{
    yc kms symmetric-key list --format json | jq -r '.[0].id'
}

#
# Secrets
#

function get_netmon_tokens_secret_id() # Get id of netmon-tokens lockbox secret
{
    yc lockbox secret get netmon-tokens --format json | jq -r '.id'
}

#
# Network
#

function get_net_name() # Get network name
{
    yc vpc network list --format json | jq -r '.[] | select (.name | test("'"$NETMON_NETWORK_PREFIX"'-*")) .name'
}

get_subnet()
{
    zone=$1
    net_name=$(get_net_name)
    yc vpc network list-subnets --format json --name $net_name | jq -r '.[] | select (.name | test("'"$net_name"'-*")) | select (.zone_id | test ("'"$zone"'")) .id'
}

function get_subnets() # Get all subnet ids
{
    for zone in $ZONES
    do
        get_subnet $zone
    done
}

get_cidr_blocks()
{
    v=$1
    net_name=$(get_net_name)
    yc vpc network list-subnets --format json --name $net_name | jq -r '.[] | select (.name | test("'"$net_name"'-*")) .'$v'_cidr_blocks[]'
}

get_v4_cidr_blocks()
{
    get_cidr_blocks v4
}

get_v6_cidr_blocks()
{
    get_cidr_blocks v6
}

get_zone_cidr_blocks()
{
    v=$1
    zone=$2
    net_name=$(get_net_name)
    yc vpc network list-subnets --format json --name $net_name | jq -r '.[] | select (.name | test("'"$net_name"'-*")) | select (.zone_id | test ("'"$zone"'")) .'$v'_cidr_blocks[]'
}

get_v4_zone_cidr_blocks()
{
    zone=$1
    get_zone_cidr_blocks v4 $zone
}

get_v6_zone_cidr_blocks()
{
    zone=$1
    get_zone_cidr_blocks v6 $zone
}

# now unused
get_net_subnet()
{
    net=$1
    subnet_idx=$2
    printf "import ipaddress\nimport sys\ns = '$net'\nif s.startswith('2a02'): net = ipaddress.IPv6Network(unicode(s))\nelse: net = ipaddress.IPv4Network(unicode(s))\nnets=list(net.subnets())\nprint(nets[$subnet_idx])" | python
}

function reserve_address() # Reserve address
{
    folder_id=$(get_folder)
    addr_req="""
folder_id: $folder_id
ipv6_address_spec:
  requirements:
    hints: [yandex-only]
"""
    ycp vpc address create "$@" -r - <<REQ
$addr_req
REQ
}

function get_reserved_address() # Get reserved address
{
    ycp vpc address list "$@" --format json | jq -r '.[0].ipv6_address.address'
}

#
# DNS
#

get_dns_zone_id()
{
    yc dns zone list --format json | jq -r '.[] | select(.public_visibility) .id'
}

function get_dns_zone() # Get netmon DNS zone
{
    zone_id=$(get_dns_zone_id)
    yc dns zone get --id $zone_id --format json | jq -r '.zone' | sed 's/.$//'
}

create_dns_record()
{
    dns_name=$1
    ipv6=$2
    zone_id=$(get_dns_zone_id)
    yc dns zone add-records --id $zone_id --record "$dns_name $DNS_TTL AAAA $ipv6"
}

delete_dns_record()
{
    dns_name=$1
    ipv6=$2
    zone_id=$(get_dns_zone_id)
    yc dns zone delete-records --id $zone_id --record "$dns_name $DNS_TTL AAAA $ipv6"
}

#
# Container registry
#

function create_cr() # Create container registry
{
    yc container registry create --name netmon
}

function get_cr() # Get container registry id
{
    yc container registry list "$@" --format json | jq -r '.[0].id'
}

function get_cr_address() # Get container registry address
{
    dns_zone=$(get_dns_zone)
    echo $dns_zone | sed 's/netmon/cr/g'
}

function delete_cr() # Delete container registry
{
    yc container registry delete --name netmon
}

#
# VMs
#

create_frontend_vms() # Create VMs for aggregator and GUI
{
    net_name=$(get_net_name)
    for zone in $ZONES
    do
        subnet_id=$(get_subnet $zone)
        yc compute instance create --name $NETMON_FRONTEND-$zone --zone $zone --cores $NETMON_FRONTEND_CORES --memory $NETMON_FRONTEND_MEMORY --create-disk size=$NETMON_FRONTEND_DISK --network-interface ipv4-address=auto,ipv6-address=auto,subnet-id=$subnet_id --metadata serial-port-enable=1 --metadata-from-file user-data=users.yaml
    done
}

get_frontend_vms() # Get frontend VM ids
{
    yc compute instance list --format json | jq -r '.[] | select (.name | test("'"$NETMON_FRONTEND"'-*")) .id'
}

delete_frontend_vms() # Delete frontend VMs
{
    vms=$(get_frontend_vms)
    for vm in $vms
    do
        disks=$(yc compute disks list --format json | jq -r '.[] | select (.instance_ids[] | contains("'"$vm"'")) .id')
        yc compute instance delete --id $vm
        for disk in $disks
        do
            yc compute disks delete --id $disk
        done
    done
}

get_vm_name()
{
    id=$1
    yc compute instance get --format json --id $id | jq -r '.name'
}

get_vm_zone()
{
    id=$1
    yc compute instance get --format json --id $id | jq -r '.zone_id'
}

get_vm_subnet()
{
    id=$1
    yc compute instance get --format json --id $id | jq -r '.network_interfaces[0].subnet_id'
}

get_vm_ipv4()
{
    id=$1
    yc compute instance get --format json --id $id | jq -r '.network_interfaces[0].primary_v4_address.address'
}

get_vm_ipv6()
{
    id=$1
    yc compute instance get --format json --id $id | jq -r '.network_interfaces[0].primary_v6_address.address'
}

#
# Managed k8s
#

function create_k8s() # Create managed k8s
{
    sa_name=$(get_active_sa)
    net_name=$(get_net_name)
    yc k8s cluster create --name netmon --regional --service-account-name $sa_name --node-service-account-name $sa_name --network-name $net_name --cluster-ipv4-range 10.0.0.0/16 --dual-stack
}

function get_k8s() # Get managed k8s id
{
    yc k8s cluster list --format json | jq -r '.[0].id'
}

function delete_k8s() # Delete managed k8s
{
    id=$(get_k8s)
    yc k8s cluster delete --id $id
}

#
# Node groups
#

get_nodes()
{
    name=$1
    yc k8s node-group list-nodes --format json --name $name | jq -r '.[].cloud_status.id'
}

create_nodes()
{
    name=$1
    cores=$2
    memory=$3
    disk=$4
    labels=$5

    cluster_id=$(get_k8s)
    locations=""
    subnets=""
    num=0
    for zone in $ZONES
    do
        subnet_id=$(get_subnet $zone)
        locations+=$(echo " --location zone=$zone")
        subnets+=$(echo ",subnets=$subnet_id")
        num=$((num + 1))
    done
    yc k8s node-group create --fixed-size $num --name $name --cores $cores --memory $memory --disk-size $disk --cluster-id=$cluster_id $locations --network-interface ipv4-address=auto,ipv6-address=auto$subnets --metadata serial-port-enable=1 --metadata-from-file ssh-keys=ssh_keys.txt --node-labels $labels

    nodes=$(get_nodes $name)
    for node in $nodes
    do
        vm_name=$(get_vm_name $node)
        zone=$(get_vm_zone $node)
        ipv6=$(get_vm_ipv6 $node)
        create_dns_record $vm_name $ipv6
        create_dns_record $name-$zone $ipv6
    done
}

delete_nodes()
{
    name=$1
    nodes=$(get_nodes $name)
    for node in $nodes
    do
        zone=$(get_vm_zone $node)
        ipv6=$(get_vm_ipv6 $node)
        delete_dns_record $name-$zone $ipv6
    done

    yc k8s node-group delete --name $name
}

get_nodes_zone_ips()
{
    name=$1
    zone=$2
    nodes=$(get_nodes $name)
    for node in $nodes
    do
        if [[ "$zone" == "$(get_vm_zone $node)" ]]; then
            ipv6=$(get_vm_ipv6 $node)
            echo $ipv6
        fi
    done
}

function create_frontend_nodes() # Create node group for aggregator and GUI
{
    create_nodes $NETMON_FRONTEND $NETMON_FRONTEND_CORES $NETMON_FRONTEND_MEMORY $NETMON_FRONTEND_DISK $NETMON_FRONTEND_LABELS
}

function get_frontend_nodes() # Get frontend node ids
{
    get_nodes $NETMON_FRONTEND
}

function get_frontend_nodes_zone_ips() # Get frontend nodes IPs in selected zone
{
    get_nodes_zone_ips $NETMON_FRONTEND $1
}

function delete_frontend_nodes() # Delete frontend node group
{
    delete_nodes $NETMON_FRONTEND
}

function create_backend_nodes() # Create node group for slicer and clickhouse
{
    create_nodes $NETMON_BACKEND $NETMON_BACKEND_CORES $NETMON_BACKEND_MEMORY $NETMON_BACKEND_DISK $NETMON_BACKEND_LABELS
}

function get_backend_nodes() # Get backend node ids
{
    get_nodes $NETMON_BACKEND
}

function get_backend_nodes_zone_ips() # Get backend nodes IPs in selected zone
{
    get_nodes_zone_ips $NETMON_BACKEND $1
}

function delete_backend_nodes() # Delete backend node group
{
    delete_nodes $NETMON_BACKEND
}

#
# Network balancer
#

get_frontend_balancer_target_group()
{
    yc load-balancer target-group get --name $NETMON_FRONTEND --format json | jq -r '.id'
}

function create_frontend_balancer() # Create balancer over frontend nodes
{
    yc load-balancer target-group create $NETMON_FRONTEND
    nodes=$(get_frontend_nodes)
    for node in $nodes
    do
        subnet_id=$(get_vm_subnet $node)
        ipv6=$(get_vm_ipv6 $node)
        yc load-balancer target-group add-targets $NETMON_FRONTEND --target subnet-id=$subnet_id,address=$ipv6
    done
    yc load-balancer network-load-balancer create $NETMON_FRONTEND --type external --target-group target-group-id=$(get_frontend_balancer_target_group),healthcheck-name=tcp-port-8080,healthcheck-tcp-port=8080 --listener name=$NETMON_FRONTEND,port=80,target-port=8080,external-ip-version=ipv6
}

function get_frontend_balancer() # Get frontend balancer
{
    yc load-balancer network-load-balancer list --format json | jq -r '.[] | select (.name | test("'"$NETMON_FRONTEND"'")) .id'
}

function delete_frontend_balancer() # Delete frontend balancer
{
    yc load-balancer network-load-balancer delete $NETMON_FRONTEND
    yc load-balancer target-group delete $NETMON_FRONTEND
}

function help() # List of all available functions
{
    grep "^function" $0
}

if [[ $# < 1 ]]; then
    s=""
    help
else
    "$@"
fi
