import logging
import time
import traceback

from .component import Component

try:
    from typing import Dict, Tuple  # noqa
except ImportError:
    pass

LOG = logging.getLogger(__name__)


class Bus(Component):
    data = {}  # type: Dict[str, dict]
    last_get = {}  # type: Dict[str, float]
    heartbeat = {}  # type: Dict[str, float]
    module_errors = {}  # type: Dict[str, Tuple[float, str]]

    def init(self):  # type: () -> None
        default_module_heartbeat = self.config.get('default_module_heartbeat')
        assert default_module_heartbeat
        with self.lock:
            self.data = {}
            self.last_get = {}
            self.heartbeat = {}
            self.module_errors = {}

            for modname, instance in self.context['modules'].items():
                self.last_get[modname] = 0.
                self.heartbeat[modname] = instance.config.get('heartbeat', default_module_heartbeat)

    def __repr__(self):  # type: () -> str
        res = 'Bus\n\n'
        res += '=Modules\n\n'
        now = time.time()
        res += 'Module\theartbeat\tdata got, s ago\n'
        with self.lock:
            for modname in self.context['modules'].keys():
                res += '%s\t%i\t%i\n' % (modname, self.heartbeat[modname], now - self.last_get[modname])
        return res

    def loop(self):  # type: () -> None
        loop_time = time.time()
        with self.lock:
            for modname, module in self.context['modules'].items():
                if loop_time - self.last_get[modname] > self.heartbeat[modname]:
                    LOG.info('getting data from %s', modname)
                    started = time.time()
                    values = []
                    try:
                        values = module.get_value()
                        self._set_module_data(modname, values)
                        self.module_errors[modname] = (time.time(), module.pop_warnings())
                    except:
                        self.module_errors[modname] = (time.time(), traceback.format_exc())
                        LOG.exception('error getting data from module %s', modname)
                    finally:
                        if not values:
                            values = []
                        LOG.info("got data: %d records from %s in %.2f secs" %
                                 (len(values) if type(values) is list else 1,
                                  modname,
                                  time.time() - started))
                        self.last_get[modname] = loop_time

    def get_data(self, timestamp):  # type: (float) -> dict
        result = {}
        with self.lock:
            for modname, get_time in self.last_get.items():
                mod_data = self.data.get(modname, None)
                if get_time >= timestamp and mod_data is not None:
                    result[modname] = mod_data
        return result

    def get_module_states(self):  # type: () -> Dict[str, Tuple[float, str]]
        with self.lock:
            return self.module_errors.copy()

    def _set_module_data(self, modname, value):  # type: (str, dict) -> None
        self.data[modname] = value

    def get_state(self):  # type: () -> str
        results = []
        with self.lock:
            mod_names = sorted(set(self.data.keys()) | set(self.last_get.keys()) | set(self.heartbeat.keys()))
            for mod_name in mod_names:
                results.append('\t'.join(map(str, [
                    mod_name, bool(mod_name in self.data),
                    self.last_get.get(mod_name, None),
                    self.heartbeat.get(mod_name, None),
                    '%s: %s' % self.module_errors.get(mod_name, None),
                ])))

        return 'Module\tGot data\tLast get\tHeartbeat\tLast error\n%s\n' % '\n'.join(results)
