import json
import logging
import random
import time
import urllib2
from contextlib import closing
from urlparse import urljoin

from .component import Component
from .util import log_time

LOG = logging.getLogger(__name__)
TIMEOUT = 5

try:
    from typing import Dict
except ImportError:
    pass


class Request(urllib2.Request):
    def get_method(self):
        return 'POST'


class OopsReporter(Component):
    path = '/api/agent/feedback?hostname=%s'
    server = ''
    last_send_time = 0
    last_check_time = 0

    def init(self):  # type: () -> None
        assert type(self.config.get('heartbeat')) in (float, int), 'no heartbeat in reporter config'
        self.server = self.config.get('server')
        assert self.server.startswith('http://') or self.server.startswith('https://'), 'invalid server url'
        now = time.time()
        heartbeat = int(self.config.get('heartbeat'))
        self.last_check_time = now - now % heartbeat + random.randint(0, heartbeat - 1)

    def __repr__(self):  # type: () -> str
        with self.lock:
            return "OopsReporter, last_check: %.4f, last_send: %.4f, heartbeat: %s" % (
                self.last_check_time, self.last_send_time, self.config['heartbeat']
            )

    def loop(self):  # type: () -> None
        now = time.time()
        with self.lock:
            heartbeat = self.config.get('heartbeat')
            if now - self.last_check_time > heartbeat:
                self.last_check_time += heartbeat * int((now - self.last_check_time) / heartbeat)
                data = self.get_data()
                if data:
                    LOG.info('got data from %s modules', len(data))
                    if self._send_data(data):
                        self.last_send_time = now
                        LOG.info("feedback sent")
                else:
                    LOG.info('no data, skipping report')

    def get_data(self):  # type: () -> list
        with self.lock:
            data = self.get_component('bus').get_data(self.last_send_time)  # type: Dict[str, dict]
        if not data:
            return []
        prepared_data = []
        for v in data.values():
            if v:
                if type(v) in (list, tuple):
                    prepared_data.extend(v)
                else:
                    prepared_data.append(v)
        return prepared_data

    @log_time
    def _send_data(self, data):  # type: (list) -> bool
        packet = {
            'hostname': self.context['hostnames'],
            'agent_version': self.context['core'].state['version'],
            'data': data,
        }
        try:
            url = urljoin(self.server, self.path % packet['hostname'][0])
            if LOG.isEnabledFor(logging.DEBUG):
                LOG.debug("sending data to '%s': %s", url, json.dumps(packet))
            else:
                LOG.info("sending data to '%s'", url)

            payload = json.dumps(packet)
            req = Request(url, payload, {
                'X-oops-agent-version': packet['agent_version'],
                'Content-Type': 'application/json; charset=utf-8'
            })
            with closing(urllib2.urlopen(req, timeout=TIMEOUT)) as response:
                ans = response.read()
            if ans:
                LOG.debug('received answer: %s', ans)
            return True
        except Exception:
            LOG.critical('Failed to send report', exc_info=True)

        return False

    def get_state(self):
        res = ''
        res += 'last_check: %.4f\n' % self.last_check_time
        res += 'last_send: %.4f\n' % self.last_send_time
        res += 'heartbeat: %s\n' % self.config.get('heartbeat')
        return res
