import errno
import json
import logging
import select
import socket
import struct

from .component import Component

LOG = logging.getLogger(__name__)

BE_INT = struct.Struct('>I')


class SocketReporter(Component):
    def __init__(self, context, config=None):
        super(SocketReporter, self).__init__(context, config)
        self.socket = None

    def init(self):  # type: () -> None
        assert self.config.get('rpc_path')
        rpc_path = self.config['rpc_path']
        if isinstance(rpc_path, int):
            self.socket = socket.socket(socket.AF_INET6)
            self.socket.bind(('::1', rpc_path))
        else:
            self.socket = socket.socket(socket.AF_UNIX)
            try:
                self.socket.bind(rpc_path)
            except socket.error as e:
                if e.errno != errno.EADDRINUSE:
                    raise
                self.socket.close()
                self.socket.bind(rpc_path)
        self.socket.listen(0)

    def __repr__(self):  # type: () -> str
        return self.__class__.__name__

    def loop(self):  # type: () -> None
        if self.socket not in select.select([self.socket], [], [], 1.0)[0]:
            return

        conn, addr = self.socket.accept()
        try:
            LOG.debug("accepted connection from %s", addr or 'unix socket')
            if conn not in select.select([conn], [], [], 30.0)[0]:
                LOG.info("no command received from socket, bye-bye")
                return
            # FIXME (torkve) replace with normal api
            cmd = conn.recv(1)
            if cmd == b'\x01':
                # send just all current stats
                data = self.get_component('bus').get_data(0)  # FIXME
            elif cmd == b'\x02':
                data = {'modules': self.get_component('bus').get_module_states()}
            else:
                LOG.warning("unknown rpc command: %r", cmd)
                return
            data = json.dumps(data, indent=4)
            packed_len = BE_INT.pack(len(data))
            conn.sendall(packed_len + data)
        finally:
            conn.close()

    def stop(self):  # type: () -> None
        self.socket.close()
        super(SocketReporter, self).stop()
