import errno
import fcntl
import logging
import os
import socket
import time

LOG = logging.getLogger(__name__)


def daemonize():
    # See http://www.erlenstar.demon.co.uk/unix/faq_toc.html#TOC16
    if os.fork():  # launch child and...
        os._exit(0)  # kill off parent
    os.setsid()
    if os.fork():  # launch child and...
        os._exit(0)  # kill off parent again.
    null = os.open('/dev/null', os.O_RDWR)
    for i in range(3):
        try:
            os.dup2(null, i)
        except OSError as e:
            if e.errno != errno.EBADF:
                LOG.critical("%s", e)
                raise
    os.close(null)


def check_pid(pid):
    """ Check For the existence of a unix pid. """
    try:
        os.kill(pid, 0)
    except OSError:
        LOG.info('pid %s is not running' % pid)
        return False
    else:
        LOG.info('pid %s is running' % pid)
        return True


def get_my_hostnames():
    names = list(set([socket.getfqdn() for _ in range(10)]))
    LOG.info("my hostnames is %s", names)
    return names


def log_time(func):
    def timer(*arg, **kw):
        t1 = time.time()
        res = func(*arg, **kw)
        t2 = time.time()
        LOG.info('## function %s done in %.3f s', func.func_name, t2 - t1)
        return res

    return timer


class PidLockFile(object):
    def __init__(self, path):
        self.path = path
        self.pidfile = None

    def is_running(self):
        if not os.path.isfile(self.path):
            return False
        with open(self.path, 'r') as f:
            try:
                pid = int(f.read())
            except:
                return False
        if check_pid(pid) and self.is_locked():
            return pid
        return False

    def is_locked(self):
        if not os.path.isfile(self.path):
            return False

        try:
            fcntl.flock(open(self.path, 'r'), fcntl.LOCK_EX | fcntl.LOCK_NB)
            self.try_unlock()
            return False
        except IOError:
            return True

    def get_age(self):
        info = os.stat(self.path)
        return time.time() - info.st_mtime

    def __enter__(self):
        pid = self.is_running()
        if pid:
            raise LockError('Already running, pid %i', pid)

        self.pidfile = open(self.path, 'a')
        try:
            fcntl.flock(self.pidfile.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError:
            logging.fatal("cannot lock - pidfile %s is locked", self.path)
            raise LockError("Can't lock pid file " + self.path)
        self.write_pid()
        return self

    def __exit__(self, *args, **kw):
        if self.pidfile and not self.pidfile.closed:
            self.try_unlock()
            try:
                self.pidfile.close()
            except IOError:
                pass
        if os.path.isfile(self.path):
            os.remove(self.path)

    def try_unlock(self):
        try:
            fcntl.flock(self.pidfile.fileno(), fcntl.LOCK_UN | fcntl.LOCK_NB)
        except IOError:
            pass

    def write_pid(self):
        self.pidfile.seek(0)
        self.pidfile.truncate()
        self.pidfile.write(str(os.getpid()))
        self.pidfile.flush()
        self.pidfile.seek(0)


class LockError(Exception):
    pass
