import json
import logging
import re
import sys

from .abstract import PlainModule, run_command, iso_date

LOG = logging.getLogger(__name__)


def get_val(s):  # type: (str) -> str
    if ':' not in s:
        return s
    return s.split(':', 2)[1].strip()


class AgentModule(PlainModule):
    def get_value(self):
        result = {}

        cpulist = []
        cpu = {}
        if self.arch.startswith('linux'):
            for s in open('/proc/cpuinfo', 'r').readlines():
                if not s.strip():
                    continue
                k, v = map(str.strip, s.strip().split(':', 1))
                if k == 'processor':
                    if cpu:
                        cpulist.append(cpu)
                    cpu = {}
                for a, b in (('physical id', 'p_id'), ('core id', 'c_id'), ('cpu cores', 'cores'), ('flags', 'flags'),
                             ('model name', 'model_name'), ('cpu MHz', 'cpu_freq')):
                    if k == a:
                        cpu[b] = v

            cpulist.append(cpu)
            names = set([a.get('model_name') for a in cpulist])
            if len(names) == 1:
                result['processorName'] = re.sub(r'\s+', ' ', names.pop())
            result['processorsCount'] = len(set([int(a.get('p_id', 0)) for a in cpulist]))
            result['coresCount'] = len(set([int(a.get('c_id', 0)) for a in cpulist]))
            result['maxFreq'] = int(max(float(a.get('cpu_freq', 0)) for a in cpulist))
            result['logicalCoresCount'] = len(cpulist)
            fl = cpulist[0]['flags'].split()
            fl.sort()
            result['bitness'] = 64 if 'lm' in fl else 32
            result['hyperThreading'] = True if 'ht' in fl else False
            result['flags'] = fl

        elif self.arch.startswith('freebsd'):
            result['processorName'] = re.sub(r'\s+', ' ', get_val(run_command('sysctl hw.model').out))
            result['bitness'] = 64 if '64' in run_command('sysctl hw.machine_arch').out else 32
            s = run_command('grep -i FreeBSD/SMP /var/run/dmesg.boot').out
            r = re.search(r'(\d+) package', s, re.M)
            if r:
                result['processorsCount'] = int(r.group(1))
            r = re.search(r'(\d+) core', s, re.M)
            if r:
                result['coresCount'] = int(r.group(1)) * result.get('processorsCount', 1)
            r = re.search(r'(\d+) SMT', s, re.M)
            if r:
                result['logicalCoresCount'] = int(r.group(1)) * result.get('coresCount', 1)

                # result['coresCount'] = get_val(run_command('sysctl hw.ncpu'))
        if result:
            result['changeTime'] = iso_date()
        return self.format_answer('cpuinfo', result) if result else None


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
