import json
import logging
import os
import sys

from .abstract import PlainModule, run_command, measure, iso_date, resource_usage

LOG = logging.getLogger(__name__)


def get_home_path():
    if os.environ.get('HOME'):
        return os.path.realpath(os.environ['HOME'].rsplit('/', 1)[0])
    return None


def is_good_mount(mp, fs):
    if mp == '/dev':
        return False
    if ':/' in fs:
        return True
    if mp == '/db':
        return True
    if mp == '/' and fs != 'rootfs':
        return True
    if fs.startswith('/') or fs.startswith('root/'):
        return True
    return False


def get_freespace(p):
    """
    Returns the number of free bytes on the drive that p is on
    """
    s = os.statvfs(p)
    return s.f_frsize * s.f_bavail, s.f_frsize * s.f_blocks


def get_df(lines=None):
    if not lines:
        lines = run_command('df -k', lines=True).out
    res = []
    try:
        prev_line = ''
        for s in lines[1:]:
            if not s.strip():
                continue
            if len(s.strip().split()) == 1:
                prev_line = s
                continue
            if s.startswith(' '):
                line = prev_line + s
                prev_line = ''
            else:
                line = s
            try:
                fs, total, used, free, usep, mp = line.split()
                if is_good_mount(mp, fs):
                    res.append({'dev': fs, 'mp': mp, 'total': int(total) * 1024, 'used': int(used) * 1024,
                                'free': int(free) * 1024})
            except:
                pass
    except:
        LOG.exception('error getting df result from %s', lines)
    return res


class AgentModule(PlainModule):

    default_config = {
        'heartbeat': 600.,
        'du_file': '/var/log/du_home.log',
        'home_location': '/home',
        'threshold_mb': 1.01
    }

    def get_value(self):
        users_mp, users_data = None, None
        if self.config.get('du_file') and os.path.isfile(self.config['du_file']):
            users_mp, users_data = self.read_du_file()

        resources = {}
        for m in get_df():
            res = {'registrationTime': iso_date(), 'name': '%s on %s' % (m['dev'], m['mp']), 'capacity': m['total'],
                   'usage': measure(m['used'])}
            usages = []
            if users_mp and m['mp'] != '/' and users_mp.startswith(m['mp']):
                usages = resource_usage(users_data)
            res['consumerToUsage'] = usages
            resources[m['mp']] = res
        result = {'name': 'DisksUsage', 'resources': resources}
        return self.format_answer('disk_usage', result)

    def read_du_file(self, lines=None):
        home_mp = get_home_path()
        if not lines:
            if not os.path.isfile(self.config['du_file']):
                return home_mp, {}
            with open(self.config['du_file'], 'r') as f:
                lines = f.readlines()
        res = {}
        threshold = self.config.get('threshold_mb', 0)
        for s in lines:
            s = s.strip().split()
            if len(s) == 2 and s[1].startswith('/') and not s[1].startswith('/var/tmp'):
                if home_mp and not os.path.realpath(s[1]).startswith(home_mp + '/'):
                    continue
                path, login = s[1].rstrip('/').rsplit('/', 1)
                try:
                    mb = int(s[0])
                    if mb > threshold:
                        res[login] = int(s[0]) * 1024 * 1024
                except:
                    pass
        return home_mp, res


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
