import json
import os
import socket
import sys
import traceback

from abstract import PlainModule

try:
    from typing import Optional
except ImportError:
    pass


class AgentModule(PlainModule):
    hostname = socket.gethostname()

    def get_size(self, path, sys_blk):  # type: (str, str) -> int
        try:
            with open(path, "r") as f:
                sector = int(f.read().strip())
                with open(os.path.join(sys_blk, "size"), "r") as s:
                    return sector * int(s.read().strip())
        except (IOError, ValueError):
            self.warnings.log(traceback.format_exc())
            return 0

    def get_type(self, sys_path_to_device):  # type: (str) -> str
        path = os.path.abspath(os.path.join(sys_path_to_device, "..", "queue", "rotational"))
        try:
            with open(path, "r") as f:
                rotational = f.read().strip()
                if rotational == "0":
                    return "SSD"
                elif rotational == "1":
                    return "HDD"
                else:
                    return "unknown"
        except IOError as e:
            # look at them at heartbeat: https://heartbeat.yandex-team.ru/oops_info
            self.warnings.log("error (errno=%s) reading file %s: %s",
                              e.errno, path, traceback.format_exc())
            return "unknown"

    def get_mounts(self, path="/dev/disk/by-label"):  # type: (str) -> dict
        mounts = {}
        if not os.path.isdir(path):
            with open('/proc/mounts', "r") as f:
                for mount in f:
                    device_name, mounted_to, _ = mount.split(" ", 2)
                    if len(mounted_to.split('/')) > 2:  # excluding dynamically created mounts /place/porto/etc/..
                        continue
                    device_name = os.path.realpath(device_name).split('/')[-1]
                    mounts[device_name] = mounted_to
            return mounts
        for device in os.listdir(path):
            path_to_device = os.readlink(os.path.join(path, device))
            name = os.path.basename(path_to_device)
            mounts[name] = device.replace("\\x2f", "/")
        return mounts

    def get_partition_uuid(self, path="/dev/disk/by-uuid"):  # type: (str) -> dict
        partition_uuid = {}
        if not os.path.isdir(path):
            return partition_uuid
        for device in os.listdir(path):
            path_to_device = os.readlink(os.path.join(path, device))
            name = os.path.basename(path_to_device)
            partition_uuid[name] = device
        return partition_uuid

    def get_disks_hw_info(self, path="/dev/disk/by-id"):  # type: (str) -> dict
        hw_info = {}
        if not os.path.isdir(path):
            return hw_info
        for device in os.listdir(path):
            path_to_device = os.readlink(os.path.join(path, device))
            name = os.path.basename(path_to_device)
            try:
                model_serial = device.split("-", 2)[1]
                if model_serial.startswith('uuid'):
                    model_serial = device.split("-", 2)[2]
                if model_serial.startswith('name'):
                    model_serial = device.split("-", 2)[2]
            except IndexError:
                model_serial = device
            hw_info[name] = model_serial
        return hw_info

    def get_value(self):  # type: () -> Optional[dict]
        manufacturer = self.get_disks_hw_info()
        mounts = self.get_mounts()
        partition_uuid = self.get_partition_uuid()

        path_by_uuid = "/dev/disk/by-uuid"
        if not os.path.isdir(path_by_uuid):
            return None
        devices = os.listdir(path_by_uuid)
        result = []
        for device in devices:
            handle_device = self.handle_device(path_by_uuid, device, manufacturer, mounts, partition_uuid)
            result.extend(handle_device)

        return self.format_answer("disks", {"disks": result}) if result else None

    def handle_device(self, path, device, disk_name_serial, mounts,
                      partition_uuid):  # type: (str, str, dict, dict, dict) -> list

        path_to_device = os.path.join(path, device)
        disk_info = {"device": path_to_device}

        if os.path.islink(path_to_device):
            path_to_device = os.path.join(path, os.readlink(path_to_device))

        device_name = os.path.basename(path_to_device)
        disk_info["name"] = device_name
        try:
            mounted_to = mounts[device_name]
        except KeyError:
            disk_info["mountPoint"] = None
            disk_info["fsSize"] = None
        else:
            disk_info["mountPoint"] = mounted_to
            try:
                statvfs = os.statvfs(mounted_to)
                disk_info["fsSize"] = statvfs.f_blocks * statvfs.f_frsize
            except OSError:
                disk_info["fsSize"] = None
        try:
            disk_info["hwInfo"] = disk_name_serial[device_name]
        except KeyError:
            disk_info["hwInfo"] = None

        try:
            disk_info["partitionUuid"] = partition_uuid[device_name]
        except KeyError:
            disk_info["partitionUuid"] = None
        try:
            device_stat = os.stat(path_to_device)
        except OSError:
            self.warnings.log(traceback.format_exc())
            return []
        device_number = "{}:{}".format(os.major(device_stat.st_rdev), os.minor(device_stat.st_rdev))
        sys_path_to_device = os.path.realpath("/sys/dev/block/{}".format(device_number))
        if 'md' not in path_to_device and 'dm' not in path_to_device:
            logical_block_size_path = os.path.abspath(
                os.path.join(sys_path_to_device, "..", "queue", "logical_block_size"))
            disk_info.update({
                'deviceNumber': device_number,
                'size': self.get_size(logical_block_size_path, sys_path_to_device),
                'type': self.get_type(sys_path_to_device),
                'slaves': [],
                'raidLevel': None,
            })
            return [disk_info]

        logical_block_size_path = os.path.abspath(os.path.join(sys_path_to_device, "queue", "logical_block_size"))
        disk_info["size"] = self.get_size(logical_block_size_path, sys_path_to_device)
        try:
            path_md_level = os.path.abspath(os.path.join(sys_path_to_device, "md", "level"))
            with open(path_md_level, "r") as f:
                disk_info["raidLevel"] = f.read().strip()
        except IOError:
            disk_info["raidLevel"] = None

        raid_slaves = []
        slaves_info = []
        for slave in os.listdir(os.path.join(sys_path_to_device, "slaves")):
            dev_path = "/dev"
            raid_slaves.append(slave)
            slaves_info.extend(
                self.handle_device(dev_path, slave, disk_name_serial, mounts, partition_uuid))

        slaves_types = {s['type'] for s in slaves_info}

        if len(slaves_types) == 1:
            disk_info["type"] = slaves_types.pop()
        elif "HDD" in slaves_types:
            disk_info["type"] = "HDD"
        else:
            disk_info["type"] = "unknown"

        disk_info["slaves"] = raid_slaves
        disk_info["deviceNumber"] = device_number

        return [disk_info] + slaves_info


if __name__ == "__main__":
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
