import logging
import os
import sys
import traceback

from .abstract import PlainModule, run_command, iso_date

try:
    from typing import List
    from .abstract import Warnings
except ImportError:
    pass

LOG = logging.getLogger(__name__)


def get_val(s):  # type: (str) -> str
    if ':' not in s:
        return s
    return s.split(':', 2)[1].strip()


def read_dmi_info(path, warnings):  # type: (str, Warnings) -> str
    try:
        with open(os.path.join('/sys/class/dmi/id/', path)) as f:
            return f.read().decode('utf-8', 'ignore').encode('utf-8').strip()
    except Exception:
        warnings.log("failed to read %s: %s", path, traceback.format_exc())
        return ""


def parse_dmi(lines):  # type: (List[str]) -> dict
    res = {}
    key = None
    handle = False
    for l in lines:
        if l.startswith('Handle '):
            handle = True
            continue
        if handle:
            handle = None
            key = l
            continue
        if ':' in l and key:
            k, v = [x.strip() for x in l.split(':', 1)]
            if v:
                res.setdefault(key, {})[k] = v
    return res


class AgentModule(PlainModule):
    def get_value(self):
        if self.arch.startswith('linux'):
            result = dict(
                system=read_dmi_info('sys_vendor', self.warnings) + ' ' + read_dmi_info('product_name', self.warnings),
                motherboard=read_dmi_info('board_vendor', self.warnings) + ' ' + read_dmi_info('board_name',
                                                                                               self.warnings),
                biosName=read_dmi_info('bios_vendor', self.warnings),
                biosVersion=read_dmi_info('bios_version', self.warnings),
                biosDate=read_dmi_info('bios_date', self.warnings),
                changeTime=iso_date())
        elif self.arch.startswith('freebsd'):
            result = {}
            s = run_command('sudo dmidecode', lines=True)
            if not s.returncode:
                d = parse_dmi(s.out)
                section = d.get('System Information', {})
                result['system'] = ' '.join([section.get('Manufacturer', ''), section.get('Product Name', '')]).strip()
                section = d.get('Base Board Information', {})
                result['motherboard'] = ' '.join([
                    section.get('Manufacturer', ''), section.get('Product Name', '')
                ]).strip()
                result['biosName'] = section.get('Vendor', '')
                result['biosRevision'] = section.get('BIOS Revision', '')
                result['biosVersion'] = section.get('Version', '')
                result['changeTime'] = iso_date()
        else:
            return None

        return self.format_answer('dmidecode', result) if result else None


if __name__ == '__main__':
    import json

    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
