import math
import time


class EWMA(object):

    def __init__(self, interval=60.):
        self.interval = interval
        self.m1_alpha = None
        self.m5_alpha = None
        self.m15_alpha = None
        self.last_window_start = None
        self.calculate_alpha()
        self.values = []
        self.m1 = None
        self.m5 = None
        self.m15 = None
        self.avg = 0
        self.min = 0
        self.max = 0
        self.last = 0

    def __repr__(self):
        res = []
        for n in ('last', 'min', 'max', 'avg', 'm1', 'm5', 'm15'):
            res.append("%s: %s" % (n, getattr(self, n)))
        return ', '.join(res)

    def calculate_alpha(self):
        self.m1_alpha = 1 - math.exp(-self.interval / 60.)
        self.m5_alpha = 1 - math.exp(-self.interval / 300.)
        self.m15_alpha = 1 - math.exp(-self.interval / 900.)
        self.last_window_start = time.time()
        self.last_window_start -= self.last_window_start % self.interval

    def update(self, val):
        if val is not None:
            now = time.time()
            self.last = val
            self.values.append((now, val))
        self._tick()

    def _tick(self):
        now = time.time()
        if now - self.last_window_start < self.interval:
            return
        num_ticks = int((now - self.last_window_start) / self.interval)
        t2 = now - now % self.interval
        t1 = t2 - self.interval

        values = [x for x in self.values if t1 <= x[0] <= t2]

        self.max = max([x[1] for x in values]) if values else 0
        self.min = min([x[1] for x in values]) if values else 0
        self.avg = sum([x[1] for x in values]) / len(values) if values else 0
        while self.values and self.values[0][0] < t2:
            self.values.pop(0)

        self.last_window_start = t2

        for _ in range(num_ticks - 1):
            self.m1 = self._update_m(self.m1, 0, self.m1_alpha)
            self.m5 = self._update_m(self.m5, 0, self.m5_alpha)
            self.m15 = self._update_m(self.m15, 0, self.m15_alpha)
        self.m1 = self._update_m(self.m1, self.avg, self.m1_alpha)
        self.m5 = self._update_m(self.m5, self.avg, self.m5_alpha)
        self.m15 = self._update_m(self.m15, self.avg, self.m15_alpha)

    def _update_m(self, m_value, value, alpha):
        if m_value is None:
            m_value = value
        else:
            m_value += alpha * (value - m_value)
        return m_value

    def dict(self):
        self._tick()
        res = {}
        for n in ('last', 'min', 'max', 'avg', 'm1', 'm5', 'm15'):
            res[n] = getattr(self, n)
        return res


if __name__ == '__main__':
    import random

    e = EWMA()

    while 1:
        time.sleep(random.randint(1, 5))
        e.update(random.randint(0, 255))
        print e.dict()
