import json
import logging
import re
import sys
import traceback
from collections import namedtuple

from .abstract import PlainModule, run_command

try:
    from typing import Dict, Optional, List  # noqa
    from .abstract import Warnings  # noqa
except ImportError:
    pass

LOG = logging.getLogger(__name__)

IfaceInfo = namedtuple('IfaceInfo', ['name', 'macAddress', 'bandwidth', 'ips', 'fwInfo', 'vlan', ])


def get_firmware_info(interface):  # type: (str) -> Optional[dict]
    if interface == 'lo':
        return None

    res = {}
    result = run_command('/sbin/ethtool -i {}'.format(str(interface)))
    if result.returncode != 0:
        return res
    bus_info = None
    for line in result.out.splitlines():
        line = line.strip()
        if line.startswith('driver'):
            res['driver'] = line[8:]
        if line.startswith('version'):
            res['version'] = line[9:]
        if line.startswith('bus-info'):
            bus_info = line[-7:]

    if bus_info:
        result = run_command('/usr/bin/lspci')
        for line in result.out.splitlines():
            if line.startswith(bus_info):
                res['modelName'] = line[29:]

    return res


def get_int_mac(interface):  # type: (str) -> Optional[str]
    mac_file = '/sys/class/net/{}/address'.format(str(interface))
    try:
        with open(mac_file, 'r') as f:
            mac_raw = f.read().strip()
        return mac_raw
    except Exception:
        LOG.warning('cannot read file {}'.format(mac_file))
    return None


def get_int_speed(interface):  # type: (str) -> Optional[int]
    if interface == 'lo':
        return None

    speed_file = '/sys/class/net/{}/speed'.format(str(interface))
    try:
        with open(speed_file, 'r') as f:
            return int(f.read())
    except Exception:
        LOG.warning('cannot read file {}'.format(speed_file))

    return None


def get_hostnames():  # type: () -> dict
    res = dict(
        name=run_command('hostname').out.strip(),
        fqdn=run_command('hostname -f').out.strip(),
        fqdns=run_command('hostname -A').out.strip().split()
    )
    return res


def get_re(reg, s):
    r = re.search(reg, s)
    if r:
        return r.group(1)
    return None


def parse_ip(warnings):  # type: (Warnings) -> List[IfaceInfo]
    try:
        lines = run_command('ip -o addr show', lines=True).out
    except OSError:
        warnings.log("failed to call `ip addr show`: %s", traceback.format_exc())
        return []

    results = {}  # type: Dict[str, IfaceInfo]
    for line in lines:
        line = line.strip()
        if not line:
            continue
        try:
            values = map(str.strip, line.split('\\'))
            m = re.match(
                r'(?P<idx>\d+):\s*(?P<iface>\S+)\s+(?P<inet>inet6?)\s+(?P<ip>\S+)\s+scope\s+(?P<scope>\S+).*',
                values[0])
            if not m:
                warnings.log('bad line in `ip addr` response: %s', line)
                continue

            ip_info = m.groupdict()
            idx = ip_info['idx']

            if idx not in results:
                # new record
                iface = ip_info['iface']
                result = IfaceInfo(name=iface,
                                   macAddress=None,
                                   bandwidth=None,
                                   ips=[],
                                   fwInfo=None,
                                   vlan=None)
                results[idx] = result
            else:
                result = results[idx]

            inet = ip_info.get('inet', None)
            ip = ip_info.get('ip', None)
            if ip and inet == 'inet' or (inet == 'inet6' and ip_info.get('scope', None) == 'global'):
                result.ips.append(ip)

        except Exception:
            warnings.log('failed to process line in `ip addr` response `%s`: %s', line, traceback.format_exc())

    try:
        lines = run_command('ip -o -d link show', lines=True).out
    except OSError:
        warnings.log("failed to call `ip addr show`: %s", traceback.format_exc())
        return results.values()

    dummies = set()
    for line in lines:
        line = line.strip()
        if not line:
            continue
        try:
            values = map(str.strip, line.split('\\'))
            m = re.match(r'(?P<idx>\d+):\s*(?P<iface>\S+):.*', values[0])
            if not m:
                warnings.log('bad line in `ip link` response: %s', line)
                continue

            ip_info = m.groupdict()
            idx = ip_info['idx']
            result = results.get(idx, None)
            if not result:
                continue
            iface = ip_info['iface']  # like `lo`, `eth0`, `ip6tnl0@NONE`, `vlan688@eth0`, `L3-0@eth1`, etc.
            if iface != result.name and not iface.startswith(result.name + '@'):
                warnings.log("inconsistent iface name in `ip`'s responses: %s / %s", iface, result.name)
                continue

            m = re.match(r'vlan protocol\s+.*\s+id\s+(?P<id>\d+)\s+\S*', values[-1])
            if m:
                result = result._replace(vlan=int(m.group('id')))
                results[idx] = result
            if values[-1] == 'dummy':
                dummies.add(idx)

        except Exception:
            warnings.log('failed to process line in `ip link` response `%s`: %s', line, traceback.format_exc())

    for idx, iface_info in results.items():
        iface = iface_info.name
        try:
            mac_addr = get_int_mac(iface)
            bandwidth = None if idx in dummies else get_int_speed(iface)
            try:
                fw_info = get_firmware_info(iface)
            except Exception:
                warnings.log("failed to get firmware info for %s: %s", iface, traceback.format_exc())
                fw_info = None

            results[idx] = iface_info._replace(
                macAddress=mac_addr,
                bandwidth=bandwidth,
                fwInfo=fw_info
            )
        except Exception:
            warnings.log('failed to enrich iface info `%s`: %s', iface, traceback.format_exc())

    return results.values()


def parse_ifconfig_bsd():  # type: () -> List[IfaceInfo]
    lines = run_command('ifconfig', lines=True).out
    res = []
    d = None
    for s in lines:
        if not s.strip():
            continue
        if s[0] not in (' ', '\t'):
            ss = s.split()
            d = IfaceInfo(name=ss[0].strip(':'), macAddress=None, bandwidth=None, ips=[], fwInfo=None, vlan=None)
            res.append(d)
        r = get_re(r'ether\s+(\S+)', s)
        if r:
            d.macAddress = r.replace(':', '').upper()
        r = get_re(r'inet (\S+)', s)
        if r:
            d.ips.append(r)
        r = get_re(r'inet6 (\S+)', s)
        if r:
            if '%' in r:
                r = r[:r.find('%')]
            d.ips.append(r)
    return res


def get_host(ip):  # type: (str) -> Optional[str]
    ip = ip.split('/')[0]
    s = run_command('host %s' % (ip,)).out
    return get_re(r'domain name pointer (\S+)\.', s)


def get_host_records(iface_infos, warnings):  # type: (List[IfaceInfo], Warnings) -> List[Dict]
    result = []
    for iface_info in iface_infos:
        for ip in iface_info.ips:
            try:
                h = get_host(ip)
            except OSError:
                warnings.log('failed to get get_host_records oserror: %s', traceback.format_exc())
                continue
            if h:
                result.append(dict(ip=ip, fqdn=h))
    return result


class AgentModule(PlainModule):
    def get_value(self):
        if self.arch.startswith('linux'):
            interfaces = parse_ip(self.warnings)
        elif self.arch.startswith('freebsd'):
            interfaces = parse_ifconfig_bsd()
        else:
            interfaces = []
        result = dict(
            hostname=get_hostnames(),
            interfaces=map(lambda i: i._asdict(), interfaces),
            ptrs=get_host_records(interfaces, self.warnings)
        )

        return self.format_answer('netinfo', result) if result else None


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
