import json
import logging
import os.path
import re
import sys
import traceback
import urllib2
from collections import OrderedDict
from contextlib import closing

from .abstract import PlainModule, run_command

try:
    from typing import Optional  # noqa
except ImportError:
    pass

LOG = logging.getLogger(__name__)

_DEFAULT_ROOT_ON_STORAGE = 'db/iss3'
_ISS_DIRECTORIES = {
    '.',
    'jobs',
    'instances',
    'resources',
    'volumes',
    'shards',
    'logs',
}
_PORTO_DIRECTORIES = {
    'porto_layers',
    'porto_storage',
    'porto_volumes',
}
_DF_OUTPUT_PATTERN = \
    r'(?P<fs>\S+)\s+(?P<size>\d+)\s+(?P<used>\d+)\s+(?P<available>\d+)\s+(?P<use>\d+%)\s+(?P<mount>\S+).*'


class AgentModule(PlainModule):

    def get_value(self):
        config = self.read_iss_agent_config()

        result = {
            'config': config,
            'storages': self.fetch_storages(config) if config else None
        }
        return self.format_answer('iss_agent', result)

    def read_iss_agent_config(self):  # type: () -> Optional[dict]
        try:
            with closing(urllib2.urlopen('http://localhost:25536/config', timeout=3)) as r:
                config = json.load(r)

            LOG.info('got iss-agent version: %s', config.get('agent', {}).get('versionInfo', {}).get('version', None))
            return config
        except Exception:
            self.warnings.log("failed to get iss configuration: %s", traceback.format_exc())

        return None

    def fetch_storages(self, config):  # type: (dict) -> dict
        root_on_storage = config.get('agent', {}).get('rootOnStorage', _DEFAULT_ROOT_ON_STORAGE)
        storage_mount_points = config.get('agent', {}).get('storageMountPoints', [])
        results = {}
        for storage_mount_point in storage_mount_points:
            storage_result = results.setdefault(storage_mount_point, OrderedDict())
            iss_dirs = [os.path.join(storage_mount_point, root_on_storage, dir_name) for dir_name in _ISS_DIRECTORIES]
            porto_dirs = [os.path.join(storage_mount_point, dir_name) for dir_name in _PORTO_DIRECTORIES]
            paths = map(os.path.abspath, filter(os.path.isdir, iss_dirs + porto_dirs))
            for path in paths:
                storage_info = self.get_storage_info(path)
                if storage_info:
                    storage_result[path] = storage_info

        return results

    def get_storage_info(self, path):  # type: (str) -> Optional[dict]
        try:
            lines = run_command(['df', path, '-P'], lines=True).out or []
        except OSError:
            self.warnings.log("failed to call `df %s`: %s", path, traceback.format_exc())
            return None

        if len(lines) != 2:
            self.warnings.log('bad out lines `df %s` response: %s', path, str(lines))
            return None
        m = re.match(_DF_OUTPUT_PATTERN, lines[-1])
        if not m:
            self.warnings.log('bad line in `df %s` response: %s', path, lines[-1])
            return None
        storage_info = m.groupdict()
        return dict(fs=storage_info['fs'])


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
