import json
import logging
import os
import sys
import traceback

from .abstract import PlainModule

try:
    from typing import Optional
except ImportError:
    pass

LOG = logging.getLogger(__name__)


class AgentModule(PlainModule):
    server_info_path = '/etc/server_info.json'
    salt_hs_state_path = '/var/run/salt/last_run_state'

    def is_salt_exist(self):  # type: () -> bool
        return os.path.isfile(self.server_info_path)

    def load_server_info(self):
        with open(self.server_info_path, 'r') as f:
            try:
                data = json.load(f)
            except Exception:
                self.warnings.log("Exception in load %s Exc: %s", self.server_info_path, traceback.format_exc())
                return None
        if not isinstance(data, dict):
            self.warnings.log("Data is not a dict, please check %s", self.server_info_path)
            return None
        # Fresh for custom_states
        try:
            if os.path.isfile(self.salt_hs_state_path):
                with open(self.salt_hs_state_path, 'r') as f_hs:
                    data_hs = json.load(f_hs)
                    data['custom_states'] = data_hs['custom_states']
        except Exception:
            self.warnings.log("Bad data in %s. Exception : %s", self.salt_hs_state_path, traceback.format_exc())

        return data

    def get_server_number(self, data):  # type: (dict) -> Optional[int]
        if 'server_number' not in data:
            return None
        # check type
        LOG.info('salt server_number : {}'.format(data['server_number']))
        return int(data['server_number'])

    def get_pillars(self, data):  # type: (dict) -> Optional[dict]
        pillar = data.get('pillar', None)
        if not pillar:
            return None
        if not isinstance(pillar, dict):
            self.warnings.log("Pillar is not a dict", self.server_info_path)
            return None
        LOG.info('salt got pillars')
        return pillar

    def get_grains(self, data):  # type: (dict) -> dict
        # check type
        return_data = {}

        if 'location' in data:
            # SAS,VLA,MAN,MSK (MSK is MYT,IVA and UGR)
            return_data['location'] = data['location']
        if 'gcfg_tag' in data:
            # Released tag in GCFG
            return_data['gcfg_tag'] = data['gcfg_tag']
        if 'gencfg' in data:
            # All groups in gencfg
            return_data['gencfg'] = data['gencfg']
        if 'custom_states' in data:
            return_data['custom_states'] = data['custom_states']

        LOG.info('salt got grains')
        return return_data

    def get_value(self):
        if not self.is_salt_exist():
            return None
        data = self.load_server_info()
        # check answer
        if data is None:
            return None
        result = {
            'server_number': self.get_server_number(data),
            'pillar': self.get_pillars(data),
            'grains': self.get_grains(data)
        }

        return self.format_answer('salt', result)


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
