import json
import logging
import os
import sys
from hashlib import md5, sha256

from .abstract import PlainModule, Warnings, iso_date, log_time

LOG = logging.getLogger(__name__)
SHARD_PATH = ('/db/BASE', '/db/iss3/shards', '/db/www/jupiter')

STATE_FILE = 'shard.state'
CONF_FILE = 'shard.conf'


def read_state_file(path, warnings):  # type: (str, Warnings) -> dict
    data = {}
    with open(path, 'r') as f:
        for line in f:
            line = line.strip()
            if not line:
                continue
            if not (':' in line):
                warnings.log('invalid line in state file "%s": %s', path, line)
                continue
            k, v = line.split(':', 1)
            data[k] = v
    return data


def convert(data):  # type: (dict) -> dict
    res = {}
    key_convert = {'generated_chksum': 'generatedChksum', 'sky-result': 'skyResult'}
    for key in ('name', 'acquire', 'generated_chksum', 'install', 'sky-result', 'place'):
        new_key = key_convert.get(key, key)
        if key in data.keys():
            res[new_key] = data.get(key)
    res['skyTries'] = int(data.get('skytries')) if data.get('skytries') else None
    res['created'] = iso_date(int(data.get('created'))) if data.get('created') else None
    return res


def read_shard(path, warnings):  # type: (str, Warnings) -> dict
    shard_data = read_state_file(os.path.join(path, STATE_FILE), warnings)
    data = convert(shard_data)
    if data.get('acquire') == 'OK' or data.get('install') == 'OK':
        data.update(get_dir_digest_and_size(path))
    return data


class AgentModule(PlainModule):

    @log_time
    def get_value(self):
        result = []
        shard_dir = False
        for base_path in SHARD_PATH:
            if not os.path.isdir(base_path):
                continue
            shard_dir = True
            for f in os.listdir(base_path):
                if not os.path.isdir(os.path.join(base_path, f)):
                    continue
                if os.path.isfile(os.path.join(base_path, f, STATE_FILE)):
                    LOG.info('found shard %s', f)
                    shard_data = {'name': f, 'place': base_path}
                    shard_data.update(read_shard(os.path.join(base_path, f), self.warnings))
                    result.append(shard_data)
                    continue
                if os.path.isfile(os.path.join(base_path, f, CONF_FILE)):
                    LOG.info('found shard %s, no shardstate', f)
                    shard_data = {'name': f, 'place': base_path, 'install': 'OK'}
                    result.append(shard_data)

        if not result:
            LOG.error('no shards found')

        if result:
            return self.format_answer('shardstates', {'shards': result})

        if shard_dir:
            return self.format_answer('shardstates', {'shards': []})

        return None


def get_dir_digest_and_size(d):  # type: (str) -> dict
    files = [x for x in os.listdir(d) if os.path.isfile(os.path.join(d, x))]
    full_size = 0
    digest = sha256()
    for fn in files:
        file_path = os.path.join(d, fn)
        size = os.lstat(file_path).st_size
        full_size += size
        file_digest = ''
        # try:
        #     file_digest = get_file_md5(file_path)
        # except:
        #     file_digest = ''
        digest.update('%s %s %s\n' % (fn, size, file_digest))

    return {'sha256': digest.hexdigest(), 'size': full_size}


def get_file_md5(fn):
    n_bytes = 1024
    d = md5()
    size = os.stat(fn).st_size
    with open(fn, 'rb') as f:
        if size > n_bytes:
            f.seek(-n_bytes, 2)
        a = f.read(n_bytes)
        d.update(a)
    return d.hexdigest()


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
