import json
import logging
import os
import re
import sys

from .abstract import PlainModule, iso_date, run_command

LOG = logging.getLogger(__name__)


def check_skynet_python_version():
    return run_command('/skynet/python/bin/python --version').err.strip().split()[1]


def get_svn_sky_version(s=None):
    if not s:
        if not os.path.isfile('/skynet/svninfo'):
            LOG.error('/skynet/svninfo not found')
            return
        with open('/skynet/svninfo', 'r') as f:
            s = f.readlines()
    for line in s:
        r = re.search(r'Revision:\s?(\d+)', line)
        if r:
            LOG.info('skynet svn revision: %s', r.group(0))
            return 'svn %s' % r.group(1)
    return None


def check_kqueue():
    cmd = ['/skynet/python/bin/python', '-c', '"from api.cqueue import Client; c=Client()"']
    ret = run_command(cmd)
    return ret.returncode == 0


def check_copier():
    cmd = ['/skynet/python/bin/python', '-c', '"from api.copier import Client; c=Client()"']
    ret = run_command(cmd)
    return ret.returncode == 0


def check_skynet_version():
    version = ''
    out = run_command('/skynet/tools/sky --version').out.strip()
    LOG.info('got skynet version: %s', out)
    r = re.search(r'(\d[\d.]+\d)', out)
    if r:
        version = r.group(1).strip('.')
    else:
        LOG.error('no version in skynet out: %s' % out)
    if not version:
        version = get_svn_sky_version()
    return version or 'no version'


class AgentModule(PlainModule):
    def get_value(self):
        result = {}
        if os.path.isdir('/skynet'):
            result['version'] = check_skynet_version()
            result['pythonVersion'] = check_skynet_python_version()
            result['cqueue'] = check_kqueue()
            result['copier'] = check_copier()

        if result:
            result['changeTime'] = iso_date()

        return self.format_answer('skynet', result) if result else None


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
