import json
import logging
import os
import re
import sys
import time

from .abstract import PlainModule, iso_date, run_command, event

LOG = logging.getLogger(__name__)


def parse_lsb():
    if not os.path.isfile('/etc/lsb-release'):
        return None
    parms = {}
    with open('/etc/lsb-release', 'r') as f:
        for l in f.readlines():
            k, v = map(lambda x: x.strip('"').strip(), l.strip().split('='))
            parms[k] = v

    if 'DISTRIB_DESCRIPTION' in parms:
        return parms['DISTRIB_DESCRIPTION']
    if 'DISTRIB_ID' in parms and 'DISTRIB_RELEASE' in parms:
        return "%s (%s)" % (parms['DISTRIB_ID'], parms['DISTRIB_RELEASE'])

    return None


class AgentModule(PlainModule):
    context = None

    def get_value(self):
        os_info = {}
        uptime = {}
        if self.arch.startswith('linux'):
            os_info = {'type': os.uname()[0], 'kernel': os.uname()[2], 'version': parse_lsb()}
            with open('/proc/uptime', 'r') as f:
                uptime_seconds = float(f.readline().split()[0])
            uptime = event('systemEvents', 'SystemStart', -uptime_seconds)

        elif self.arch.startswith('freebsd'):
            os_info = {'type': os.uname()[0], 'version': os.uname()[2]}
            s = run_command('sysctl kern.boottime').out
            r = re.search(r'sec = (\d+)', s)
            if r:
                uptime = event('systemEvents', 'SystemStart', int(r.group(1)))

        result = []
        if os_info:
            result.append(self.format_answer('osinfo', os_info))
        if uptime:
            result.append(self.format_answer('uptime', uptime))
        if self.context and self.context.get('core'):
            result.append(self.format_answer('agent', dict(
                version=self.context['core'].state.get('version', '?'),
                startTime=iso_date(self.context['core'].state['start_time'])
            )))
        return result


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    agent = AgentModule(sys.platform)
    agent.config['agent_version'] = '5.5'
    agent.config['start_time'] = time.time()
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
