import json
import logging
import re
import sys
import traceback
import urllib2
from contextlib import closing

from .abstract import PlainModule, run_command

try:
    from typing import Dict, Optional
    from .abstract import Warnings
except ImportError:
    pass

LOG = logging.getLogger(__name__)


def check_python_version():  # type: () -> Optional[str]
    try:
        v = run_command(['python', '--version']).err.strip().split()[1]
        LOG.info('got python version: %s', v)
        return v
    except:
        LOG.exception('Failed to read python version')

    return None


def check_porto_version():  # type: () -> Optional[str]
    try:
        res = run_command('/usr/sbin/portoctl -v')
        s = res.out + res.err
        r = re.search(r'client:\s+v?(\S+)', s)
        if r:
            version = r.group(1)
            LOG.info('got porto version: %s', version)
            return version
    except:
        LOG.exception('Failed to read porto version')

    return None


def check_hwwatcher_version():  # type: () -> Optional[str]
    try:
        s = run_command('/usr/sbin/hw_watcher --version').out
        r = re.search(r'hw_watcher\s+(\S+)', s)
        if r:
            version = r.group(1)

            LOG.info('got hw_watcher version: %s', version)
            return version
    except:
        LOG.exception('Failed to get hw_watcher version')

    return None


def check_issagent_version():  # type: () -> Optional[str]
    try:
        with closing(urllib2.urlopen('http://localhost:25536/version', timeout=3)) as r:
            version = json.loads(r.read())

        LOG.info('got iss-agent version: %s', version)
        return version['version']
    except:
        LOG.exception('Failed to get iss agent version')

    return None


def check_yasmagent_version():  # type: () -> Optional[str]
    try:
        with closing(urllib2.urlopen('http://localhost:11003/version/', timeout=3)) as r:
            version = r.read()

        LOG.info('got yasmagent version: %s', version)
        return version
    except:
        LOG.exception('Failed to get yasm agent version')

    return None


def get_dpkg_packages_version(warnings, excludes=None):  # type: (Warnings, Optional[list]) -> Dict[str, str]
    return_data = {}
    try:
        report = list()
        for s in run_command('dpkg -l', lines=True).out:
            pkg = s.split(None, 3)
            if pkg[0] != 'ii':
                continue
            if excludes and pkg[1] in excludes:
                continue
            return_data[pkg[1]] = pkg[2].strip()
            report.append("%s=%s" % (pkg[1], pkg[2].strip()))
        LOG.debug("got " + ", ".join(report))
        return return_data
    except:
        warnings.log('Failed to get dpkg all versions: %s', traceback.format_exc())

    return return_data


class AgentModule(PlainModule):

    def get_value(self):
        result = {
            'python': check_python_version(),
            'hwwatcher': check_hwwatcher_version(),
            'issagent': check_issagent_version(),
            'yasmagent': check_yasmagent_version(),
            'porto': check_porto_version()
        }

        if self.arch.startswith('linux'):
            result['packages'] = get_dpkg_packages_version(warnings=self.warnings)

        return self.format_answer('versions', result)


if __name__ == '__main__':
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
