import json
import logging
import os
import socket
import sys
import traceback

from .abstract import PlainModule, run_command

LOG = logging.getLogger(__name__)

YA_NETCONFIG_DIR = "/run/network/ya-netconfig/"
NETCONFIG_FILENAME = "ya_netconfig_state.json"
STATES_FILENAME_MASK = "%s.state"


class AgentModule(PlainModule):

    def _normalize_ip(self, address_string):
        try:
            return socket.inet_ntop(socket.AF_INET6, socket.inet_pton(socket.AF_INET6, address_string))
        except (socket.error, TypeError):
            self.warnings.log("Invalid address string: %s", address_string)
            return address_string

    def _get_json_from_file(self, dirname, filename):
        try:
            with open(os.path.join(dirname, filename)) as f:
                return json.load(f)
        except IOError:
            self.warnings.log("failed to open %s: %s", filename, traceback.format_exc())
        except json.JSONDecodeError:
            self.warnings.log("failed to parse %s: %s", filename, traceback.format_exc())
        return {}

    def _check_files(self):
        netconfig_state = self._get_json_from_file(YA_NETCONFIG_DIR, NETCONFIG_FILENAME)
        results = []
        for iface in set([x['interface'] for x in netconfig_state]):
            state = self._get_json_from_file(YA_NETCONFIG_DIR, "%s.state" % (iface,))

            try:
                results.append(
                    dict(
                        ip=self._normalize_ip(state['bb_ip']),
                        os_ip=None,
                        type='backbone',
                        verified=False,
                        iface=iface
                    )
                )
            except KeyError:
                self.warnings.log("Key error while parsing %s: %s", iface, traceback.format_exc())

            for vlan, vlan_info in state['fastbone'].items():
                try:
                    ip = vlan_info['mtn_global'] if 'host64' in vlan_info else vlan_info['fb_ip']
                    fb_iface = vlan_info['fb_iface']
                    if fb_iface in results:
                        self.warnings.log("Duplicate vlan in %s.state: %s", iface, fb_iface)
                    results.append(dict(
                        ip=self._normalize_ip(ip.split('/')[0]),
                        os_ip=None,
                        type='fastbone',
                        verified=False,
                        iface=fb_iface,
                        vlan=vlan,
                        parent=iface
                    ))
                except KeyError:
                    self.warnings.log("Key error while parsing vlan%s in %s: %s", vlan, iface, traceback.format_exc())
        return results

    def _verify(self, ya_netconfig_data):
        try:
            os_result = [x for x in run_command('ip -6 -o a', lines=True).out if x]
        except OSError:
            self.warnings.log("Unable to run ip -6 -o a: %s", traceback.format_exc())
            return ya_netconfig_data

        os_ifaces = {}
        for line in os_result:
            try:
                iface = line.split()
                if len(iface) < 6:
                    continue

                iface_name = iface[1]
                iface_scope = iface[5]
                ip = iface[3].split('/')[0]

                if iface_scope != 'global':
                    continue

                os_ifaces[iface_name] = ip
            except KeyError:
                self.warnings.log("Key error while parsing ip results %s", traceback.format_exc())

        for netconfig_state in ya_netconfig_data:
            netconfig_iface_name = netconfig_state['iface']
            if netconfig_iface_name in os_ifaces:
                netconfig_state['os_ip'] = self._normalize_ip(os_ifaces[netconfig_iface_name])
                netconfig_state['verified'] = netconfig_state['ip'] == netconfig_state['os_ip']
                if not netconfig_state['verified']:
                    self.warnings.log("IP mismatch on interface %s", netconfig_iface_name)
            else:
                self.warnings.log("Interface not present in OS: %s", netconfig_iface_name)
        return ya_netconfig_data

    def get_value(self):
        result = {'ifaces': self._verify(self._check_files())}
        return self.format_answer('ya_netconfig_state', result) if result else None


if __name__ == "__main__":
    logging.basicConfig(level='INFO')
    print json.dumps(AgentModule(sys.platform).get_value(), indent=4)
