from __future__ import absolute_import

import importlib
import sys
import types

sys.dont_write_bytecode = True


class LibraryProxyLoader(object):
    def __init__(self, from_root, to_root):
        self.from_root = from_root
        self.to_root = to_root

    def load_original_module(self, fullname):
        name = self.from_root + fullname[len(self.to_root):]
        mod = importlib.import_module(name)
        return mod

    def load_module(self, fullname):
        if fullname in sys.modules:
            return sys.modules[fullname]

        name = self.to_root + fullname[len(self.from_root):]
        mod = importlib.import_module(name)
        sys.modules[fullname] = mod
        return mod


class LibraryModProxy(types.ModuleType):
    def __init__(self, name, from_root, to_root):
        super(LibraryModProxy, self).__init__(name)
        self.__loader__ = LibraryProxyLoader(from_root, to_root)

    def __getattr__(self, name):
        mod = self.__loader__.load_original_module(self.__name__)
        return getattr(mod, name)

    def __dir__(self):
        mod = self.__loader__.load_module(self.__name__)
        return dir(mod)


def make_proxy(from_root, to_root):
    for modname in sorted(list(globals()['__loader__'].memory)):
        m = modname[:-9] if modname.endswith('.__init__') else modname
        if m == from_root or m.startswith(from_root + '.'):
            proxyname = to_root + m[len(from_root):]
            proxy = LibraryModProxy(proxyname, from_root, to_root)
            parent, shortname = proxyname.rsplit('.', 1)
            setattr(importlib.import_module(parent), shortname, proxy)
            sys.modules[proxyname] = proxy


if __name__ == '__main__':
    if getattr(sys, 'is_standalone_binary', False):
        """
        We can remap modules like that:

        import pkg_resources

        pkg_resources.require = lambda *args, **kwargs: None

        make_proxy('kernel.util', 'ya.skynet.util')
        make_proxy('library', 'ya.skynet.library')
        """
        pass

    from ya.infra.oops.agent import oops_agent

    oops_agent.main()
