import requests

from infra.orly.proto import orly_pb2


def failure(error, message, code):
    return orly_pb2.Status(status='Failure', error=error, message=message, code=code)


class Orly(object):
    PROD_URL = 'https://orly.yandex-team.ru'
    HEADERS = {
        'Content-Type': 'application/x-protobuf',
        'Accept': 'application/x-protobuf',
        'User-Agent': 'orlyctl',
    }

    def __init__(self, url=PROD_URL):
        url = url.rstrip('/')
        self.url = url + '/rest/'

    def _call(self, method, request, response):
        url = self.url + method + '/'
        try:
            resp = requests.post(url,
                                 data=request.SerializeToString(),
                                 headers=self.HEADERS,
                                 timeout=10)
        except Exception as e:
            return None, failure('CallFailed', 'request failed: {}'.format(e), 500)
        if resp.status_code == 200:
            # Try to parse response
            try:
                response.MergeFromString(resp.content)
            except Exception as e:
                return None, failure('InvalidResponse',
                                     'failed to parse response: {}'.format(e),
                                     code=500)
            return response, None
        # Try to parse error response
        status = orly_pb2.Status()
        try:
            status.MergeFromString(resp.content)
        except Exception as e:
            return None, failure('InvalidResponse',
                                 'bad code={}, failed to parse response: {}'.format(resp.status_code, e),
                                 resp.status_code)
        return None, status

    def create_rule(self, request):
        response = orly_pb2.CreateRuleResponse()
        return self._call('CreateRule', request, response)

    def get_rule(self, request):
        response = orly_pb2.GetRuleResponse()
        return self._call('GetRule', request, response)

    def start_operation(self, request):
        response = orly_pb2.OperationResponse()
        return self._call('StartOperation', request, response)

    def delete_rule(self, request):
        response = orly_pb2.DeleteRuleResponse()
        return self._call('DeleteRule', request, response)
