package client

import (
	"bytes"
	"crypto/tls"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/golang/protobuf/proto"

	pb "a.yandex-team.ru/infra/orly/proto"
)

const URLProduction = "https://orly.yandex-team.ru/"

type Client struct {
	url        string
	headers    http.Header
	httpClient http.Client
}

func NewClient(url string) *Client {
	if url[len(url)-1] == '/' {
		url = url[:len(url)-1]
	}
	url += "/rest/"
	return &Client{
		headers: http.Header{
			"Accept":       []string{headerValueProtobuf},
			"Content-Type": []string{headerValueProtobuf},
			"User-Agent":   []string{"infra/orly/go/client"},
		},
		url: url,
		httpClient: http.Client{
			Timeout: 5 * time.Second,
			// Do not verify certificate (just to keep working in case of emergency)
			Transport: &http.Transport{
				TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
			},
		},
	}
}

// do creates requests to awacs, and serves responses from it
func (c *Client) do(method string, req, resp proto.Message) error {
	buf, err := proto.Marshal(req)
	if err != nil {
		return err
	}
	httpReq, err := http.NewRequest("POST", c.url+method+"/", bytes.NewBuffer(buf))
	if err != nil {
		return err
	}
	httpReq.Header = c.headers
	httpResp, err := c.httpClient.Do(httpReq)
	if err != nil {
		return err
	}
	defer func() {
		_ = httpResp.Body.Close()
	}()
	buf, err = ioutil.ReadAll(httpResp.Body)
	if err != nil {
		return fmt.Errorf("failed to read response: %s", err.Error())
	}
	if httpResp.StatusCode != 200 {
		// Need to deserialize Status
		status := &pb.Status{}
		err := proto.Unmarshal(buf, status)
		if err != nil {
			return fmt.Errorf("failed to parse response: %s", err.Error())
		}
		return fmt.Errorf("bad response code: %d, message: %s", httpResp.StatusCode, status.Error)
	}
	return proto.Unmarshal(buf, resp)
}

func (c *Client) CreateRule(req *pb.CreateRuleRequest) (*pb.CreateRuleResponse, error) {
	resp := &pb.CreateRuleResponse{}
	return resp, c.do("CreateRule", req, resp)
}

func (c *Client) GetRule(req *pb.GetRuleRequest) (*pb.GetRuleResponse, error) {
	resp := &pb.GetRuleResponse{}
	return resp, c.do("GetRule", req, resp)
}

func (c *Client) StartOperation(req *pb.OperationRequest) (*pb.OperationResponse, error) {
	resp := &pb.OperationResponse{}
	return resp, c.do("StartOperation", req, resp)
}

func (c *Client) DeleteRule(req *pb.DeleteRuleRequest) (*pb.DeleteRuleResponse, error) {
	resp := &pb.DeleteRuleResponse{}
	return resp, c.do("DeleteRule", req, resp)
}
