package ctl

import (
	"a.yandex-team.ru/infra/orly/go/orly/client"
	"a.yandex-team.ru/infra/orly/go/orly/yamlutil"
	pb "a.yandex-team.ru/infra/orly/proto"
	"encoding/json"
	"fmt"
	"github.com/golang/protobuf/jsonpb"
	"github.com/golang/protobuf/proto"
	"io/ioutil"
)

var marshaller = jsonpb.Marshaler{
	EnumsAsInts:  false,
	EmitDefaults: false,
	Indent:       "    ",
	OrigName:     true,
	AnyResolver:  nil,
}

func protoFromFile(path string, m proto.Message) error {
	yamlFile, err := ioutil.ReadFile(path)
	if err != nil {
		return err
	}
	var data map[string]interface{}
	err = yaml.Unmarshal(yamlFile, &data)
	if err != nil {
		return err
	}
	j, err := json.Marshal(data)
	if err != nil {
		return err
	}
	err = jsonpb.UnmarshalString(string(j), m)
	if err != nil {
		return err
	}
	return nil
}

func GetRule(client *client.Client, id string) (string, error) {
	rule, err := client.GetRule(&pb.GetRuleRequest{
		RuleId: id,
	})
	if err != nil {
		return "", fmt.Errorf("can not get rule.\nerr: %s", err)
	}
	data, err := marshaller.MarshalToString(rule)
	if err != nil {
		return "", fmt.Errorf("response marshalling error.\nerr: %s", err)
	}
	return data, nil
}

func ApplyRule(client *client.Client, ruleFilename string) (string, error) {
	var rule pb.Rule
	err := protoFromFile(ruleFilename, &rule)
	if err != nil {
		return "", fmt.Errorf("can not read rule from %s.\nerr: %s", ruleFilename, err)
	}
	resp, err := client.CreateRule(&pb.CreateRuleRequest{
		Rule: &rule,
	})
	if err != nil {
		return "", fmt.Errorf("can not apply rule.\nerr: %s", err)
	}
	data, err := marshaller.MarshalToString(resp)
	if err != nil {
		return "", fmt.Errorf("response marshalling error.\nerr: %s", err)
	}
	return data, nil
}

func DeleteRule(client *client.Client, id string) (string, error) {
	_, err := GetRule(client, id)
	if err != nil {
		return "", err
	}
	_, err = client.DeleteRule(&pb.DeleteRuleRequest{
		RuleId: id,
	})
	if err != nil {
		return "", fmt.Errorf("can not delete rule.\nerr: %s", err)
	}
	resp := fmt.Sprintf("Successfully delete rule %s", id)
	return resp, err
}

func StartOperation(client *client.Client, ruleFilename string) (string, error) {
	var req pb.OperationRequest
	err := protoFromFile(ruleFilename, &req)
	if err != nil {
		return "", fmt.Errorf("can not read operation from %s.\nerr: %s", ruleFilename, err)
	}
	resp, err := client.StartOperation(&req)
	if err != nil {
		return "", fmt.Errorf("can not process operation.\nerr: %s", err)
	}
	data, err := marshaller.MarshalToString(resp)
	if err != nil {
		return "", fmt.Errorf("response marshalling error.\nerr: %s", err)
	}
	return data, nil
}
