import collections
import http
import logging

import flask

from google.protobuf import json_format

from infra.orly.lib import climit
from infra.orly.lib import limited_semaphore
from infra.orly.lib import storage
from infra.orly.proto import orly_pb2

from . import validation
from . import selector

log = logging.getLogger('api')

CONTENT_TYPE_JSON = 'application/json'

SUCCESS = "Success"
FAILURE = "Failure"

# StatusReason is an enumeration of possible failure causes.  Each StatusReason
# must map to a single HTTP status code, but multiple reasons may map
# to the same HTTP status code.


# Means the server has declined to indicate a specific reason.
# The details field may contain other information about this error.
# Status code 500.
CODE_UNKNOWN = http.HTTPStatus.INTERNAL_SERVER_ERROR
REASON_UNKNOWN = ""
# Means the server can be reached and understood the request, but requires
# the user to present appropriate authorization credentials (identified by the WWW-Authenticate header)
# in order for the action to be completed. If the user has specified credentials on the request, the
# server considers them insufficient.
CODE_UNAUTHORIZED = http.HTTPStatus.UNAUTHORIZED
REASON_UNAUTHORIZED = "Unauthorized"
# Means the server can be reached and understood the request, but refuses
# to take any further action.  It is the result of the server being configured to deny access for some reason
# to the requested resource by the client.
CODE_FORBIDDEN = http.HTTPStatus.FORBIDDEN
REASON_FORBIDDEN = "Forbidden"
# Means one or more resources required for this operation
# could not be found.
CODE_NOT_FOUND = http.HTTPStatus.NOT_FOUND
REASON_NOT_FOUND = "NotFound"
# Means the resource you are creating already exists.
CODE_ALREADY_EXISTS = http.HTTPStatus.CONFLICT
REASON_ALREADY_EXISTS = "AlreadyExists"
# Means the requested operation cannot be completed
# due to a conflict in the operation. The client may need to alter the
# request. Each resource may define custom details that indicate the
# nature of the conflict.
CODE_CONFLICT = http.HTTPStatus.CONFLICT
REASON_CONFLICT = "Conflict"
# Means that the request itself was invalid, because the request
# doesn't make any sense, for example deleting a read-only object.  This is different than
# StatusReasonInvalid above which indicates that the API call could possibly succeed, but the
# data was invalid.  API calls that return BadRequest can never succeed.
CODE_BAD_REQUEST = http.HTTPStatus.BAD_REQUEST
REASON_BAD_REQUEST = "BadRequest"


def message_from_req(m, req):
    if req.content_type == 'application/x-protobuf':
        m.MergeFromString(req.get_data())
    else:
        json_format.Parse(req.get_data(), m, ignore_unknown_fields=True)


def err_bad_request(message):
    return None, orly_pb2.Status(
        status=FAILURE,
        error=REASON_BAD_REQUEST,
        message=message,
        code=CODE_BAD_REQUEST,
    )


def err_not_found(message):
    return None, orly_pb2.Status(
        status=FAILURE,
        error=REASON_NOT_FOUND,
        message=message,
        code=CODE_NOT_FOUND,
    )


def err_internal(message):
    return None, orly_pb2.Status(
        status=FAILURE,
        error=REASON_UNKNOWN,
        message=message,
        code=CODE_UNKNOWN,
    )


def err_conflict(message):
    return None, orly_pb2.Status(
        status=FAILURE,
        error=REASON_CONFLICT,
        message=message,
        code=CODE_CONFLICT,
    )


def err_limit_reached(message):
    return None, orly_pb2.Status(
        status=FAILURE,
        error='LimitReached',
        message=message,
        code=429)


def make_response(m, status, content_type, response_cls=flask.Response):
    if content_type == 'application/x-protobuf':
        body = m.SerializeToString()
    else:
        body = json_format.MessageToJson(m)
    return response_cls(body, status=status, content_type=content_type)


class Ctx(object):
    @classmethod
    def from_request(cls, req):
        user_ip = req.access_route[0] if req.access_route else req.remote_addr
        return cls(user_ip)

    def __init__(self, remote_addr, timeout=10.0):
        self.remote_addr = remote_addr
        self.timeout = timeout


def get_content_type(request):
    if request.headers.get('Accept', 'application/json') == 'application/x-protobuf':
        return 'application/x-protobuf'
    return 'application/json'


STATUS_LIMIT_REACHED = orly_pb2.Status(status=FAILURE,
                                       error='LimitReached',
                                       message='Max in-flight limit reached',
                                       code=429)

STATUS_BAD_METHOD = orly_pb2.Status(status=FAILURE,
                                    error='BadMethod',
                                    message='Only POST supported',
                                    code=CODE_BAD_REQUEST)


class ApiHandler(object):
    """
    Handler should return (response, status):
        * if response is None: return status
        * if response is not None: return response
    """

    def __init__(self, request_type, max_in_flight=0):
        self.request_type = request_type
        self.limit = climit.CLimit(max_in_flight) if max_in_flight else None

    def __call__(self, method):
        m_name = method.__name__
        success_counter = '{}_ok'.format(m_name)
        fail_counter = '{}_fail'.format(m_name)
        time_hgram = '{}_time'.format(m_name)

        def handler(obj):
            t = obj.registry.get_histogram(time_hgram).timer()
            r = flask.request
            content_type = get_content_type(r)
            if self.limit is not None and not self.limit.add():
                obj.registry.get_counter(fail_counter).inc()
                t.stop()
                return make_response(STATUS_LIMIT_REACHED,
                                     status=STATUS_LIMIT_REACHED.code,
                                     content_type=content_type)
            try:
                if r.method != 'POST':
                    obj.registry.get_counter(fail_counter).inc()
                    return make_response(STATUS_BAD_METHOD,
                                         status=STATUS_BAD_METHOD.code,
                                         content_type=content_type)
                m = self.request_type()
                try:
                    message_from_req(m, r)
                except Exception as e:
                    obj.registry.get_counter(fail_counter).inc()
                    st = orly_pb2.Status(status=FAILURE,
                                         error=REASON_BAD_REQUEST,
                                         message=str(e),
                                         code=CODE_BAD_REQUEST)
                    return make_response(st, st.status, content_type)
                ctx = Ctx.from_request(flask.request)
                # Call user provided handler
                try:
                    resp, status = method(obj, m, ctx)
                except Exception as e:
                    obj.registry.get_counter(fail_counter).inc()
                    log.exception(e)
                    st = orly_pb2.Status(status=FAILURE,
                                         error=REASON_UNKNOWN,
                                         message=str(e),
                                         code=CODE_UNKNOWN)
                    return make_response(st, st.status, content_type)
                if status is not None:
                    obj.registry.get_counter(success_counter).inc()
                    return make_response(status, status=status.code, content_type=content_type)
                obj.registry.get_counter(success_counter).inc()
                return make_response(resp, status=200, content_type=content_type)
            finally:
                if self.limit is not None:
                    self.limit.done()
                t.stop()

        handler.__name__ = m_name
        return handler


class ApiServer(object):
    SELECTOR_FAIL_COUNTER = 'handle_operation_selector_fail'
    FORBIDDEN_COUNTER = 'handle_operation_request_forbidden'
    ALLOW_COUNTER = 'handle_operation_request_allow'

    def __init__(self, store, registry, sched_cache, max_queue_len=30):
        """
        :type store: infra.orly.lib.storage.Storage
        :type registry: infra.orly.lib.metrics.Registry
        :type sched_cache: infra.orly.lib.sched.ScheduleCache
        """
        self.storage = store
        self.registry = registry
        self.sched_cache = sched_cache
        self.start_op_m = collections.defaultdict(lambda: limited_semaphore.LimitedQueueSemaphore(max_queue_len))

    @ApiHandler(orly_pb2.OperationRequest)
    def handle_operation_request(self, req, ctx):
        op = orly_pb2.Operation()
        op.meta.id = req.id
        op.meta.labels.update(req.labels)
        op.spec.rule = req.rule
        op.spec.remote_addr = ctx.remote_addr
        err = validation.validate_operation(op)
        if err is not None:
            return err_bad_request(err)
        op_str = '{}/{}'.format(op.spec.rule, op.meta.id)
        # Limit max queue length and acquire lock after,
        # thus decreasing potential transaction conflicts down the road.
        m = self.start_op_m[op.spec.rule]
        t = self.registry.get_histogram('queue_waiting_time', tags=(('tier', req.rule),)).timer()
        err = m.acquire()
        if err is not None:
            return err_limit_reached('rule "{}": {}'.format(op.spec.rule, err))
        t.stop()
        try:
            log.info('Starting operation %s...', op_str)
            # Get rule and check if it allows.
            # We can get conflicts (other transaction succeeded while we tried it)
            # because we are using optimistic concurrency, let's try several times
            # if we failed starting operation
            for _ in range(3):
                rule = self.storage.get_rule(op.spec.rule, ctx.timeout)
                if rule is None:
                    return err_not_found('rule "{}" not found'.format(op.spec.rule))
                err = selector.match(rule.spec.selector, op)
                if err is not None:
                    self.registry.get_counter(self.SELECTOR_FAIL_COUNTER, tags=(('tier', req.rule),)).inc()
                    return err_conflict(err)
                err = self.sched_cache.get_policy(rule).add(rule, op)
                if err is not None:
                    self.registry.get_counter(self.FORBIDDEN_COUNTER, tags=(('tier', req.rule),)).inc()
                    log.info('Operation %s failed: %s', op_str, err)
                    return err_conflict(err)
                err = self.storage.start_operation(op, rule, timeout=ctx.timeout)
                if err is None:
                    self.registry.get_counter(self.ALLOW_COUNTER, tags=(('tier', req.rule),)).inc()
                    log.info('Operation %s succeeded', op_str)
                    return orly_pb2.OperationResponse(operation=op), None
                elif err == storage.Storage.ERR_OP_IN_PROGRESS:
                    # Do not retry this error as it will not succeed, it is no a race between
                    # orly instances trying to add new operation
                    break
            self.registry.get_counter(self.FORBIDDEN_COUNTER, tags=(('tier', req.rule),)).inc()
            log.info('Operation %s failed: %s', op_str, err)
            return err_conflict(err)
        finally:
            m.release()

    @ApiHandler(orly_pb2.CreateRuleRequest, max_in_flight=5)
    def handle_create_rule(self, req, ctx):
        rule = req.rule
        err = validation.validate_rule(rule)
        if err is not None:
            return err_bad_request(err)
        rule.meta.revision = 0
        rule.status.Clear()
        log.info('Creating/updating rule:\n%s', rule)
        rule, err = self.storage.put_rule(rule, ctx.timeout)
        if err is not None:
            return err_conflict(err)
        return orly_pb2.CreateRuleResponse(rule=rule), None

    @ApiHandler(orly_pb2.GetRuleRequest, max_in_flight=5)
    def handle_get_rule(self, req, ctx):
        if not req.rule_id:
            return err_bad_request('No rule id')
        rule = self.storage.get_rule(req.rule_id, ctx.timeout)
        if rule is None:
            return err_not_found('rule "{}" not found'.format(req.rule_id))
        return orly_pb2.GetRuleResponse(rule=rule), None

    @ApiHandler(orly_pb2.DeleteRuleRequest, max_in_flight=5)
    def handle_delete_rule(self, req, ctx):
        if not req.rule_id:
            return err_bad_request('No rule id')
        if not self.storage.delete_rule(req.rule_id, ctx.timeout):
            return err_internal('Failed to delete rule {}'.format(req.rule_id))
        return orly_pb2.DeleteRuleResponse(), None


def init_api(app, storage, reg, sched_oracle):
    server = ApiServer(storage, reg, sched_oracle)
    app.add_url_rule('/rest/StartOperation/',
                     view_func=server.handle_operation_request,
                     methods=['POST'])
    app.add_url_rule('/rest/CreateRule/',
                     view_func=server.handle_create_rule,
                     methods=['POST'])
    app.add_url_rule('/rest/GetRule/',
                     view_func=server.handle_get_rule,
                     methods=['POST'])
    app.add_url_rule('/rest/DeleteRule/',
                     view_func=server.handle_delete_rule,
                     methods=['POST'])
