import logging
import random
import threading

log = logging.getLogger('gc')


class OperationCollector(threading.Thread):
    """
    Runs periodically, collecting outdated operations.
    """

    def __init__(self, storage, get_policy, min_period=10, max_period=45):
        super(OperationCollector, self).__init__(name='operation_collector')
        self.storage = storage
        self.get_policy = get_policy
        self.min_period = min_period
        self.max_period = max_period
        self._must_stop = threading.Event()

    def _next_interval(self):
        return random.randint(self.min_period, self.max_period)

    def run(self):
        while 1:
            timeout = self._next_interval()
            try:
                if self._must_stop.wait(timeout):
                    break
            except KeyboardInterrupt:
                break
            log.debug('Running operation...')
            try:
                ok, failed = self.storage.collect_stale_operations(self.get_policy)
            except Exception:
                log.exception('Failed to run gc...')
            else:
                log.info('Collected: %s, failed: %s', ok, failed)
        log.info('Stopped.')

    def stop(self, timeout=30):
        self._must_stop.set()
        self.join(timeout)
