"""
Simple metrics for YASM. Non thread-safe because of gevent.
"""
import json
import time


class Timer(object):
    __slots__ = ['hgram', '_start']

    def __init__(self, hgram):
        self.hgram = hgram
        self._start = time.time()

    def stop(self):
        self.hgram.observe(time.time() - self._start)


class Counter(object):
    def __init__(self, name, tags=(), v=0):
        self.name = name
        if not tags:
            self._fmt = name + '_dhhh'
        else:
            tags = ';'.join('{}={}'.format(k, v) for k, v in tags)
            self._fmt = '{};{}_dhhh'.format(tags, name)
        self._v = v

    def inc(self, d=1):
        self._v += d

    def fmt(self):
        return [self._fmt, self._v]

    def __str__(self):
        return str(self.fmt())


class Gauge(object):
    def __init__(self, name, v=0):
        self.name = name
        self._fmt = '{}_ahhh'.format(name)
        self._v = v

    def set(self, v):
        self._v = v

    def fmt(self):
        return [self._fmt, self._v]


class Histogram(object):
    """
    Simple histogram counting observed values.
    """
    DEFAULT_BUCKETS_SEC = (.01, .025, .05, .1, .25, .5, .75, 1.0, 2.5, 5.0, 7.5, 10.0)

    def __init__(self, name, buckets=None, tags=()):
        self.name = name
        if not tags:
            self._fmt = name + '_hgram'
        else:
            tags = ';'.join('{}={}'.format(k, v) for k, v in tags)
            self._fmt = '{};{}_hgram'.format(tags, name)
        if not buckets:
            buckets = Histogram.DEFAULT_BUCKETS_SEC
        if len(buckets) > 50:
            raise ValueError('Too many buckets: {}'.format(len(buckets)))
        buckets = [float(b) for b in buckets]
        if buckets != sorted(buckets):
            # This is probably an error on the part of the user,
            # so raise rather than sorting for them.
            raise ValueError('Buckets not in sorted order: {}'.format(buckets))
        self._buckets = [[b, 0] for b in reversed(buckets)]

    def timer(self):
        return Timer(self)

    def observe(self, amount):
        for b in self._buckets:
            if amount >= b[0]:
                b[1] += 1
                break

    def fmt(self):
        return [self._fmt, [i for i in reversed(self._buckets)]]


class Registry(object):
    def __init__(self):
        self._counters = {}
        self._gauges = {}
        self._histograms = {}

    def get_counter(self, name, tags=()):
        key = (name, tags)
        c = self._counters.get(key)
        if c is None:
            c = Counter(name, tags)
            self._counters[key] = c
        return c

    def get_gauge(self, name):
        g = self._gauges.get(name)
        if g is None:
            g = Gauge(name)
            self._gauges[name] = g
        return g

    def get_histogram(self, name, buckets=None, tags=()):
        key = (name, tags)
        h = self._histograms.get(key)
        if h is None:
            h = Histogram(name, buckets, tags=tags)
            self._histograms[key] = h
        return h

    def format(self):
        rv = [
            c.fmt() for c in self._counters.values()
        ]
        rv.extend(g.fmt() for g in self._gauges.values())
        rv.extend(h.fmt() for h in self._histograms.values())
        return json.dumps(rv)
