"""
We have a hierarchy of policies per O'RLY rule -
  * rule.spec
  * week day
  * interval of the day

There are two aspects for every rule:
    * Find policy appropriate for this time of day.
    * Call policy.add() to check if we can add operation and modify rule.
"""
import time

from . import timeutil


class Policy(object):
    """
    Interface for all rule policies.

    Policy can be non-trivial, i.e. having state more complicated
    than simple in_progress counter.
    """

    def add(self, rule, op):
        """
        Adds operation as active for the rule, updating rule's status.

        :rtype: unicode or None
        """
        raise NotImplementedError

    def done(self, rule, op):
        """
        Removes operation from active, updating rule's status.
        """
        raise NotImplementedError


class MaxAllowedPolicy(Policy):
    """
    Simple policy with only parameter - max_allowed and
    only one attribute in status - in_progress counter.
    """

    def __init__(self, max_allowed):
        self.max_allowed = max_allowed

    def add(self, rule, op):
        if self.max_allowed == 0:
            return 'no operations allowed at the moment'
        if rule.status.in_progress >= self.max_allowed:
            return 'too many operations in progress'
        rule.status.in_progress += 1
        op.meta.creation_time.GetCurrentTime()
        op.status.in_progress.status = 'True'
        op.status.in_progress.last_transition_time.MergeFrom(op.meta.creation_time)
        return None

    def done(self, rule, op):
        if rule.status.in_progress > 0:
            rule.status.in_progress -= 1
        return None


class DayInterval(object):
    @classmethod
    def from_hhmm(cls, begin, end, policy):
        return cls(timeutil.hh_mm_to_minutes(begin),
                   timeutil.hh_mm_to_minutes(end),
                   policy)

    def __init__(self, begin, end, policy):
        self.begin = begin
        self.end = end
        self.policy = policy


class DaySchedule(object):
    """
    Set of non overlapping intervals of minutes of the day (begin, end)
    each holding a policy.
    """

    def __init__(self, intervals, default):
        intervals.sort(key=lambda x: x.begin)
        self.intervals = intervals
        self.default = default

    def get_policy(self, minutes):
        # Simple linear search though the intervals (upper bound is 10)
        # We could go with binary, but there is no need - not many intervals for the day
        for i in self.intervals:
            if i.begin <= minutes <= i.end:
                # We've got a match
                return i.policy
            if i.begin > minutes:
                # Current interval begins later - no need to check further
                return self.default
        return self.default


class WeekSchedule(object):
    CACHE_TIME = 5.0

    def __init__(self, days):
        self.days = days
        # We want to avoid calling datetime and searching for policy
        # every time we want to start operation, thus we cache response for 5 seconds.
        self._next = time.time() + self.CACHE_TIME
        self._p = self._do_get_policy()

    def _do_get_policy(self):
        day, minutes = timeutil.weekday_and_minutes()
        return self.days[day].get_policy(minutes)

    def get_policy(self):
        now = time.time()
        if now > self._next:
            self._p = self._do_get_policy()
            self._next = now + self.CACHE_TIME
        return self._p


def make_schedule(rule):
    top = MaxAllowedPolicy(rule.spec.max_allowed)
    none = MaxAllowedPolicy(0)
    # Special case for rules without any day policies
    if not rule.spec.schedule.enable_days:
        days = []
        for i in range(7):
            days.append(DaySchedule([], default=top))
        return WeekSchedule(days)
    days = [[] for _ in range(7)]  # Temporary holder for intervals
    for d in rule.spec.schedule.enable_days:
        # Pick max_allowed from top or from day rule
        # If day rule max allowed is zero - take from top
        p = MaxAllowedPolicy(d.max_allowed or top.max_allowed)
        days[d.day].append(DayInterval.from_hhmm(d.begin, d.end, p))
    for i in range(7):
        days[i] = DaySchedule(days[i], default=none)
    return WeekSchedule(days)


class ScheduleCache(object):
    def __init__(self):
        self._sched = {}

    def get_policy(self, rule):
        sched, version = self._sched.get(rule.meta.id, (None, None))
        if sched is None or version != rule.meta.version:
            sched = make_schedule(rule)
            self._sched[rule.meta.id] = (sched, rule.meta.version)
        return sched.get_policy()
