import logging
import traceback

import flask
import jinja2

from google.protobuf import json_format

log = logging.getLogger('ui')

INDEX_TMPL = jinja2.Template("""
<ul>
    <li><a href="/ui/rules/">List all rules</a></li>
</ul>
""")

RULES_TMPL = jinja2.Template("""
<ul>
{% for r in rules %}
    <li><a href="/ui/rules/{{ r.meta.id }}/">Rule {{ r.meta.id }}</a></li>
{% endfor %}
</ul>
""")

RULE_TMPL = jinja2.Template("""
<h3>Rule {{rule.meta.id}}</h3>
    <ul>
        <li><a
        target="_blank"
        href="https://yasm.yandex-team.ru/template/panel/orly-ops/op={{rule.meta.id}}/">Metrics in YASM</a></li>
        <li><a href="/ui/rules/{{ rule.meta.id }}/operations/">Operations</a></li>
    </ul>
    <div>
    <pre
            style='font-family: Menlo,Monaco,Consolas,"Courier New",monospace;
                   border-radius: 4px; border: 1px solid transparent;
                   background-color: #E0E0E0; border-color: #424242;'>{{ rule_json }}</pre>
    </div>
""")

OPERATIONS_TMPL = jinja2.Template("""
<h3>Current operations for {{ rule_id }}</h3>
<ul>
    {% for op in operations %}
    <li><code>{{ json_format(op) }}</code></li>
    {% endfor %}
</ul>
""")

ERROR_TMPL = jinja2.Template("""
<h3>ERROR: {{ error }}</h>
<pre style="border-radius: 4px; border: 1px solid transparent; border-color: red;">{{ traceback }}</pre>
""")


def error_not_found(msg):
    return flask.Response(msg, status=404)


def error_bad_request(msg):
    return flask.Response(msg, status=400)


class Ctx(object):
    @classmethod
    def from_request(cls, args, req):
        return cls(args, req.remote_addr)

    def __init__(self, args, remote_addr):
        self.args = args
        self.remote_addr = remote_addr

    def get(self, item):
        return self.args.get(item)


class UiHandler(object):
    def __init__(self):
        pass

    def __call__(self, method):
        def handler(obj, **kwargs):
            ctx = Ctx.from_request(kwargs, flask.request)
            try:
                return method(obj, ctx)
            except Exception as e:
                log.exception('error during ui request')
                tb = traceback.format_exc()
                return flask.Response(ERROR_TMPL.render(error=e, traceback=tb), status=500)

        handler.__name__ = method.__name__
        return handler


class UiServer(object):
    """
    Simple UI server.
    """

    def __init__(self, storage):
        self.storage = storage

    @UiHandler()
    def to_ui(self, _):
        return flask.redirect('/ui/')

    @UiHandler()
    def index(self, _):
        return INDEX_TMPL.render()

    @UiHandler()
    def list_rules(self, _):
        rules = self.storage.list_rules()
        rules.sort(key=lambda r: r.meta.id)
        body = RULES_TMPL.render(rules=rules)
        return flask.Response(body)

    @UiHandler()
    def get_rule(self, ctx):
        rule_id = ctx.get('rule_id')
        if not rule_id:
            return error_bad_request('Empty rule provided')
        rule = self.storage.get_rule(rule_id)
        if rule is None:
            return error_not_found('Rule "{}" not found'.format(rule_id))
        body = RULE_TMPL.render(rule_json=json_format.MessageToJson(rule,
                                                                    including_default_value_fields=True),
                                rule=rule)
        return flask.Response(body)

    @UiHandler()
    def list_operations(self, ctx):
        rule_id = ctx.get('rule_id')
        ops = self.storage.list_operations(rule_id)
        ops.sort(key=lambda o: o.meta.id)
        body = OPERATIONS_TMPL.render(json_format=json_format.MessageToJson,
                                      rule_id=rule_id,
                                      operations=ops)
        return flask.Response(body)


def init_ui(app, storage):
    server = UiServer(storage)
    app.add_url_rule('/', view_func=server.to_ui)
    app.add_url_rule('/ui/', view_func=server.index)
    app.add_url_rule('/ui/rules/', view_func=server.list_rules)
    app.add_url_rule('/ui/rules/<rule_id>/', view_func=server.get_rule)
    app.add_url_rule('/ui/rules/<rule_id>/operations/', view_func=server.list_operations)
