import re

from infra.orly.proto import orly_pb2

from . import timeutil

RULE_ID_RE = re.compile(r'^[a-z]+[a-z0-9\-_]+@?$')
# Took from https://stackoverflow.com/questions/106179/regular-expression-to-match-dns-hostname-or-ip-address
OPERATION_ID_RE = re.compile(r"^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]*[a-zA-Z0-9])\.)"
                             r"*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9\-]*[A-Za-z0-9])$")
SECONDS_IN_HOUR = 60 * 60
MAX_DURATION_SEC = SECONDS_IN_HOUR
MIN_DURATION_SEC = 30
MAX_ALLOWED = 3000


def day_enum_to_string(day):
    return orly_pb2.DayRule.Day.Name(day)


def validate_hh_mm(hhmm):
    pos = hhmm.find(':')
    if pos == -1:
        return 'invalid HH:MM format in "{}"'.format(hhmm)
    hh, mm = hhmm[:pos], hhmm[pos + 1:]
    if not hh.isdigit():
        return 'invalid HH "{}" - must be digit'.format(hh)
    hh = int(hh)
    if hh < 0 or hh > 23:
        return 'invalid HH "{}" - must be between 00 and 23'.format(hh)
    if not mm.isdigit():
        return 'invalid MM "{}" - must be digit'.format(mm)
    mm = int(mm)
    if mm < 0 or mm > 59:
        return 'invalid MM "{}" - must be between 00 and 59'.format(mm)


def validate_day_intervals(intervals):
    if not intervals:
        return None
    if len(intervals) > 10:
        return 'too many intervals: {}, max - 10'.format(len(intervals))
    intervals.sort(key=lambda x: x.begin)
    # Check that intervals do not overlap
    c = intervals[0]
    for i in intervals[1:]:
        if c.end > i.begin:
            # Previous ends after next starts - overlap
            return 'overlapping intervals'
        c = i
    return None


def validate_day_rule(day_rule):
    err = validate_hh_mm(day_rule.begin)
    if err is not None:
        return 'invalid .begin for {}: {}'.format(day_rule.day, err)
    err = validate_hh_mm(day_rule.end)
    if err is not None:
        return 'invalid .end for {}: {}'.format(day_rule.day, err)
    begin = timeutil.hh_mm_to_minutes(day_rule.begin)
    end = timeutil.hh_mm_to_minutes(day_rule.end)
    if begin > end:
        return '.begin({}) > .end({})'.format(day_rule.begin, day_rule.end)
    if day_rule.max_allowed > MAX_ALLOWED:
        return '.max_allowed={} > {}'.format(day_rule.max_allowed, MAX_ALLOWED)
    if day_rule.max_allowed < 0:
        return '.max_allowed={} < 0'.format(day_rule.max_allowed)
    return None


def validate_rule_schedule(schedule):
    if not len(schedule.enable_days):
        return None
    if len(schedule.enable_days) > 70:
        return 'too many days rules={}, max=70'.format(len(schedule.enable_days))
    day_intervals = [[] for _ in range(7)]
    for day_rule in schedule.enable_days:
        err = validate_day_rule(day_rule)
        if err is not None:
            return 'invalid day rule for "{}": {}'.format(day_enum_to_string(day_rule.day), err)
        day_intervals[day_rule.day].append(day_rule)
    for i, intervals in enumerate(day_intervals):
        err = validate_day_intervals(intervals)
        if err is not None:
            return 'invalid intervals for "{}": {}'.format(day_enum_to_string(i), err)
    return None


def validate_selector(selector, path='spec.selector'):
    if len(selector.match_in) > 5:
        return 'len({}.match_in) == {} is too long, max=5'.format(path, len(selector.match_in))
    for i, m_in in enumerate(selector.match_in):
        if not m_in.key:
            return 'empty key in {}.match_in[{}]'.format(path, i)
        if len(m_in.key) > 64:
            return 'len({}.match_in[{}]) == {} is too long, max=64'.format(path, i, len(m_in.key))
        if len(m_in.values) == 0:
            return 'empty {}.match_in[{}].values'.format(path, i)
        if len(m_in.values) > 5:
            return 'len({}.match_in[{}].values) == {} is too long, max=5'.format(path, i, len(m_in.values))
        for j, v in enumerate(m_in.values):
            if not v:
                return 'empty v[{}] in {}.match_in[{}]'.format(j, path, i)
            if len(v) > 20:
                return 'len(v[{}]) == {} in {}.match_in[{}], max=20'.format(j, len(v), path, i)
    return None


def validate_rule_id(id_, path='meta.id'):
    if not id_:
        return 'no {} specified'.format(path)
    if len(id_) > 128:
        return 'len({})={} is too large, max=128'.format(path, len(id_))
    if len(id_) < 3:
        return 'len({})={} is too short, min=3'.format(path, len(id_))
    if RULE_ID_RE.match(id_) is None:
        return '{}="{}" must match {}'.format(path, id_, RULE_ID_RE.pattern)
    return None


def validate_rule(rule):
    err = validate_rule_id(rule.meta.id)
    if err is not None:
        return err
    if rule.spec.max_allowed > MAX_ALLOWED:
        return 'spec.max_allowed={} > {}'.format(rule.spec.max_allowed, MAX_ALLOWED)
    if rule.spec.max_allowed < 0:
        return 'spec.max_allowed={} < 0'.format(rule.spec.max_allowed)
    if rule.spec.duration.seconds > MAX_DURATION_SEC:
        return 'spec.duration={}s > {}s'.format(rule.spec.duration.seconds,
                                                MAX_DURATION_SEC)
    if rule.spec.duration.seconds < MIN_DURATION_SEC:
        return 'spec.duration={}s < {}s'.format(rule.spec.duration.seconds,
                                                MIN_DURATION_SEC)
    err = validate_rule_schedule(rule.spec.schedule)
    if err is not None:
        return 'spec.schedule is invalid: {}'.format(err)
    if rule.spec.HasField('selector'):
        err = validate_selector(rule.spec.selector)
        if err is not None:
            return err
    return None


def validate_operation_id(id_, path='meta.id'):
    if not id_:
        return 'No {} specified'.format(path)
    if len(id_) > 128:
        return 'len({})={} is too large, max=128'.format(path, len(id_))
    if len(id_) < 3:
        return 'len({})={} is too short, min=3'.format(path, len(id_))
    if OPERATION_ID_RE.match(id_) is None:
        return '{}="{}" must match {}'.format(path, id_, RULE_ID_RE.pattern)
    return None


def validate_operation(op):
    err = validate_operation_id(op.meta.id)
    if err is not None:
        return err
    err = validate_rule_id(op.spec.rule, path='spec.rule')
    if err is not None:
        return err
    return None
