import threading
import uuid
import argparse
import time
import sys

from google.protobuf import json_format

from infra.orly.client import Orly
from infra.orly.proto import orly_pb2

ORLY_LOCAL_URL = 'http://localhost:8080'

STATUSES = {'Succeeded': 0}
STATUSES_LOCK = threading.Lock()


class Rule:
    def __init__(self, rule_id, max_allowed, duration):
        self.id = rule_id
        self.max_allowed = max_allowed
        self.duration = duration


def init_argparse(parser):
    parser.add_argument('--url',
                        default=ORLY_LOCAL_URL,
                        help="O'RLY server url. Default: %(default)s")
    parser.add_argument('-c', '--count', help='Count of operation requests per thread', type=int, required=True)
    parser.add_argument('--max_allowed',
                        help='Max allowed parallels operation. Default: %(default)s',
                        type=int,
                        default=100)
    parser.add_argument('--duration', help='Operation duration in seconds. Default: %(default)s', type=int, default=30)
    parser.add_argument('--rule-id', help='Rule id. Default: %(default)s', type=str, default='test')
    parser.add_argument('-p', '--parallels', help='Parallel count', type=int, default=1)
    parser.add_argument('-s', '--sleep',
                        help='Sleep between operations in seconds. Default: %(default)s',
                        type=float,
                        default=0)


def create_pb_rule(m, rule):
    """
    :type m infra.orly.proto.orly_pb2
    :type rule Rule
    """
    try:
        json_format.ParseDict(
            {
                'meta': {
                    'id': rule.id
                },
                'spec': {
                    'duration': '{}s'.format(rule.duration),
                    'maxAllowed': rule.max_allowed
                }
            }, m)
    except Exception as e:
        return 'failed to create Rule: {}'.format(e)
    return None


def create_rule(o_client, rule):
    """
    :type o_client infra.orly.client.Orly
    :type rule Rule
    """
    pb_rule = orly_pb2.Rule()
    err = create_pb_rule(pb_rule, rule)
    if err:
        return None, err
    return o_client.create_rule(orly_pb2.CreateRuleRequest(rule=pb_rule))


def send_operation(o_client, count, name):
    """
    :type o_client infra.orly.client.Orly
    :type count int
    :type name str
    """
    for _ in range(count):
        op = orly_pb2.OperationRequest()
        op.id = str(uuid.uuid4())
        op.rule = name
        resp, status = o_client.start_operation(op)
        with STATUSES_LOCK:
            if resp:
                STATUSES['Succeeded'] += 1
            else:
                if status.message not in STATUSES:
                    STATUSES[status.message] = 0
                STATUSES[status.message] += 1


if __name__ == '__main__':
    parser = argparse.ArgumentParser('orly shooter')
    init_argparse(parser)
    args = parser.parse_args(sys.argv[1:])

    print '=== Starting creating rule ==='
    o_client = Orly(args.url)
    rule = Rule(args.rule_id, args.max_allowed, args.duration)
    resp, err = create_rule(o_client, rule)
    if err:
        print '=== Failed to create rule with id: "{}" ==='.format(rule.id)
        raise Exception(err)
    print '=== Successfully created rule with id: "{}" ==='.format(rule.id)

    print '=== Starting {} operations per {} threads ==='.format(args.count, args.parallels)
    ts = []
    for e in range(args.parallels):
        ts.append(threading.Thread(
            target=lambda: send_operation(o_client, args.count, rule.id)
        ))
    for t in ts:
        time.sleep(args.sleep)
        t.start()
    for t in ts:
        t.join()
    print '=== Finished {} operations with statuses: {} ==='.format(args.count * args.parallels, STATUSES)
