import argparse
import logging
import sys

import etcd3
from gevent import pywsgi
import flask

from infra.orly.lib import api
from infra.orly.lib import gc
from infra.orly.lib import ui
from infra.orly.lib import sched
from infra.orly.lib import sockutil
from infra.orly.lib import storage
from infra.orly.lib import metrics
from infra.orly.lib import wsgiutil

log = logging.getLogger('main')


def failure(message, *args, **kwargs):
    log.error(message, *args, **kwargs)
    raise SystemExit(1)


def setup_logging(level):
    logging.basicConfig(format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
                        level=level,
                        stream=sys.stdout)


def parse_args(argv):
    p = argparse.ArgumentParser('orly', description='Orly configuration service\n'
                                                    'https://wiki.yandex-team.ru/runtime-cloud/orly/#about')
    p.add_argument('-l', '--log-level',
                   choices=['DEBUG', 'INFO', 'ERROR'],
                   default='DEBUG',
                   )
    p.add_argument('-p', '--port', type=int,
                   default=8080)
    p.add_argument('-m', '--gc-max-period', type=int, default=30,
                   dest='gc_max_period', help='Max GC interval in seconds (default: %(default)s)')
    return p.parse_args(argv)


def add_util_handlers(app, registry):
    app.add_url_rule('/ping', view_func=lambda: 'OK')
    app.add_url_rule('/stat', view_func=registry.format)


def main():
    args = parse_args(sys.argv[1:])
    setup_logging(args.log_level)
    # Initialize objects that don't raise and
    # don't need to be closed/stopped
    reg = metrics.Registry()
    app = flask.Flask('orly')
    add_util_handlers(app, reg)
    cache = sched.ScheduleCache()
    log.info('Connecting to etcd...')
    etcd = etcd3.Etcd3Client(timeout=30)
    # Check connection state by getting some key
    try:
        etcd.get('/')
    except Exception as e:
        failure('Failed to connect to etcd: %s', e)
    store = storage.Storage(etcd)
    gstore = storage.GeventFriendlyStorage(store)
    api.init_api(app, gstore, reg, cache)
    ui.init_ui(app, gstore)
    collector = gc.OperationCollector(store, cache.get_policy,
                                      max_period=args.gc_max_period)
    log.info('Binding [::]:%d for HTTP server...', args.port)
    sock, err = sockutil.create_server_socket('::', args.port)
    if err is not None:
        failure('Failed to init server socket: %s', err)
    # Use gevent http server
    server = pywsgi.WSGIServer(sock, app, handler_class=wsgiutil.WSGIHandler)
    # Everything that should be closed, must be under this try block
    try:
        # For now run GC on every instance
        log.info('Starting garbage collector...')
        collector.start()
        log.info('Starting HTTP server...')
        server.serve_forever()
    except KeyboardInterrupt:
        pass
    finally:
        log.info('Stopping garbage collector..')
        collector.stop()
        log.info('Stopping HTTP server...')
        server.stop(5)
        sock.close()
    log.info('Done.')


if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        failure('Unexpected error: %s', e, exc_info=True)
