import logging
import sys
import os
import time
import socket
import signal

from infra.oxcart.proto import cluster_pb2

from infra.oxcart.lib import interface
from infra.oxcart.lib import awacsutil
from infra.oxcart.lib import spawner, cluster, envoy_grpc
from infra.oxcart.lib import its as its_
import infra.oxcart.lib.config as config_


log = logging.getLogger('main')

""" 'f' tag is a special tag, in which instancectl sends name of service.
    ITS can use this tag, to determine in which service it should deploy controls.
    We are currently using random generated name of service, due to:
    1. basicly, we don't need in control files.
    2. we don't want to intersect with some other services;
    3. we need to have control on all oxcart/envoy instances;
"""
ITS_SERVICE = 'KvglQLPDsx9JF'


def sigterm_handler(sig, frame):
    logging.info('Got SIGTERM, need to cleanup dummy interface before exit')
    # TODO: configurable iface name
    interface.Interface().delete_iface()
    sys.exit(0)


def main(config_path):

    signal.signal(signal.SIGTERM, sigterm_handler)

    logging.basicConfig(format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
                        level='DEBUG', stream=sys.stdout)

    dummy_iface = interface.Interface()
    dummy_iface.delete_iface()  # Drop previous interface state

    logging.info(config_path)
    config = config_.get_config(config_path)
    logging.info(config)
    cs = cluster.ClusterState(config)
    egrpc = envoy_grpc.EnvoyGRPC(config, cs)
    its = its_.ITS(ITS_SERVICE)

    s = spawner.Spawner(config)

    proxy_check_dl = 0
    cluster_refresh_dl = 0
    its_refresh_dl = 0

    dummy_iface.delete_iface()  # Drop previous interface state (in case oxcart was killed)
    while True:

        now = time.time()
        if now > its_refresh_dl:

            its_refresh_dl = now + config.its_check_period

            is_allowed = its.is_allowed()
            if is_allowed:
                # TODO: Move to function
                dummy_iface.create_iface()  # Create new one
                for backend in cs.cluster_config.backends:
                    logging.info('BACKEND: {}'.format(backend))
                    if backend.type == cluster_pb2.Backend.AWACS:
                        try:
                            # TODO: migrate to new api, when its done
                            # ----
                            # backend.awacs.namespace dose not always resolved to proper ip
                            # for example, there is more than one l3 balancer,
                            # so we need to use 'listen_addr' from config.
                            # listen_ip = socket.getaddrinfo(backend.awacs.namespace, None, socket.AF_INET6)[0][-1][0]
                            listen_ip = backend.listen_addr
                            logging.info('listen_ip: {}'.format(listen_ip))
                            dummy_iface.add_address(listen_ip)
                        except socket.gaierror:
                            logging.info('Could not resolve {}'.format(backend.awacs.namespace))

                    elif backend.type == cluster_pb2.Backend.STATIC:
                        dummy_iface.add_address(backend.listen_addr)

                if now > proxy_check_dl:
                    proxy_check_dl = now + (config.proxy_check_period if s.ensure() else 1)

                if now > cluster_refresh_dl:
                    cluster_refresh_dl = now + (config.cluster_refresh_period if cs.refresh() else 1)

            else:
                dummy_iface.delete_iface()

        time.sleep(1)


if __name__ == '__main__':
    if sys.argv < 2 or not os.path.exists(sys.argv[1]):
        log.error('No config file specified!')
        sys.exit(1)
    try:
        main(sys.argv[1])
    except KeyboardInterrupt:
        pass
