import ipaddr
import logging
from pyroute2 import IPRoute


class Interface(object):
    # 'dummy0' is default ifnterface for current SLB addresses,
    # We must keep it as it is for safety, so use some other name for our dummy iface.
    def __init__(self, name='dummy515', type='dummy'):
        self._name = name
        self._type = type
        self._iface_index = None

    def is_exists(self):
        try:
            with IPRoute() as ipr:
                iface_index = ipr.link_lookup(ifname=self._name)[0]
        except IndexError:
            logging.info('Device {} don\'t exists!'.format(self._name))
            self._iface_index = None
            return False

        # Update index if needed
        if self._iface_index is None or self._iface_index != iface_index:
            self._iface_index = iface_index
        logging.info('Device {} exists'.format(self._name))
        return True

    def create_iface(self):
        if self.is_exists():
            return self._iface_index

        logging.info('Trying to create device {}'.format(self._name))
        try:
            with IPRoute() as ipr:
                ipr.link('add', ifname=self._name, kind=self._type)
                self._iface_index = ipr.link_lookup(ifname=self._name)[0]
        except Exception as e:
            logging.info(e)
        else:
            logging.info('Created {}'.format(self._name))

    def delete_iface(self):
        logging.info('Delete device {}'.format(self._name))

        if not self.is_exists():
            return

        try:
            logging.info('Deleting...')
            with IPRoute() as ipr:
                ipr.link('del', ifname=self._name)
                self._iface_index = ipr.link_lookup(ifname=self._name)[0]
        except Exception as e:
            logging.info(e)
        else:
            logging.info('Deleted {}'.format(self._name))

    def get_current_addresses(self, **additional_filters):
        addresses = set()
        filters = {'index': self._iface_index}
        if additional_filters:
            filters.update(additional_filters)
        try:
            with IPRoute() as ipr:
                for ip_addr in ipr.get_addr(**filters):
                    address = ip_addr.get_attr('IFA_ADDRESS')
                    prefixlen = int(ip_addr.get('prefixlen'))
                    canonized_address = str(ipaddr.IPAddress(address))  # canonize
                    addresses.add((canonized_address, prefixlen))
        except Exception as e:
            logging.info('Some problems with getting addresses')
            logging.info(e)
            raise
        else:
            return addresses

    def is_addr_on_iface(self, addr, prefixlen=128):
        current_addreses = self.get_current_addresses()
        if (addr, prefixlen) in current_addreses:
            return True
        return False

    def add_address(self, addr, prefixlen=128):
        addr = str(ipaddr.IPAddress(addr))  # canonize

        if self.is_addr_on_iface(addr, prefixlen):
            logging.info('Address {}/{} is already on iface {}'.format(addr, prefixlen, self._name))
            return
        try:
            with IPRoute() as ipr:
                ipr.addr('add', index=self._iface_index, address=addr, prefixlen=prefixlen)
        except Exception as e:
            logging.info('Could not add address {}/{}'.format(addr, prefixlen))
            logging.info(e)

    def del_address(self, addr, prefixlen=128):
        addr = str(ipaddr.IPAddress(addr))  # canonize
        if not self.is_addr_on_iface(addr, prefixlen):
            logging.info('There is no {}/{} on iface {}'.format(addr, prefixlen, self._name))
            return

        try:
            with IPRoute() as ipr:
                ipr.addr('del', index=self._iface_index, address=addr, prefixlen=prefixlen)
        except Exception as e:
            logging.info('Could not del address {}/{}'.format(addr, prefixlen))
            logging.info(e)
