import subprocess
import logging
import porto
import os
import prctl
import ctypes
import ctypes.util
import pwd
import signal
from infra.oxcart.proto import config_pb2

log = logging.getLogger('spawner')

LIBC_NAME = ctypes.util.find_library('c')
LIBC = ctypes.CDLL(LIBC_NAME)
LIBC.__errno_location.restype = ctypes.POINTER(ctypes.c_int)

# from linux/prctl.h
PR_CAP_AMBIENT = 47
PR_CAP_AMBIENT_RAISE = 2

# from linux/capability.h
CAP_NET_BIND_SERVICE = 10


class Spawner(object):


    def __init__(self, config):
        if config.runtime_type == config_pb2.Config.PORTO:
            self.porto = porto.Connection()
            self.porto_ct = None
            self.ct_name = 'proxy'
            self.config = config

        elif config.runtime_type == config_pb2.Config.CHILD:
            self.config = config
            self.child = None
            self.envoy_user = config.envoy_user
            self.envoy_file_mode = 0o7755
            self.envoy_capabilites = 'cap_net_bind_service=+ep'  # to be fixed
        else:
            raise NotImplementedError('Not supported runtime type')

    # @staticmethod
    def set_envoy_env(self):
        prctl.cap_inheritable.net_bind_service = True
        prctl.cap_permitted.net_bind_service = True

        pr = LIBC.prctl
        pr.argtypes = [
            ctypes.c_int,
            ctypes.c_ulong,
            ctypes.c_ulong,
            ctypes.c_ulong,
            ctypes.c_ulong,
        ]

        result_pr = pr(
            ctypes.c_int(PR_CAP_AMBIENT),
            ctypes.c_ulong(PR_CAP_AMBIENT_RAISE),
            ctypes.c_ulong(CAP_NET_BIND_SERVICE),
            ctypes.c_ulong(0),
            ctypes.c_ulong(0),
        )

        if result_pr < 0:
            log.error('PRLIMIT SYSCALL: {}'.format(LIBC.__errno_location().contents))
            raise Exception()

        prctl.securebits.no_setuid_fixup = True

        os.setuid(pwd.getpwnam(self.envoy_user).pw_uid)

        prctl.set_pdeathsig(signal.SIGTERM)

    def spawn_child(self):
        try:
            binary = os.path.abspath(self.config.proxy_binary_path)

            # os.chmod(os.path.realpath(binary), self.envoy_file_mode)
            # subprocess.check_call(['/sbin/setcap', self.envoy_capabilites, os.path.realpath(binary)])  # to be fixed

            config = os.path.abspath(self.config.proxy_config_path)
            self.child = subprocess.Popen(
                [binary, '--concurrency', '4', '-c', config],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                preexec_fn=self.set_envoy_env
            )
            return self.child
        except Exception as err:
            log.info(err)

    def spawn_porto(self):
        try:
            self.porto_ct = self.porto.Find(self.ct_name)
            self.porto_ct.Destroy()
        except:
            pass

        try:
            self.porto_ct = self.porto.CreateWeakContainer(self.ct_name)
        except:
            return

        cmdline = self.config.proxy_cmdline_template.format(
            config=os.path.abspath(self.config.proxy_config_path),
            binary=os.path.abspath(self.config.proxy_binary_path)
        )
        self.porto_ct.SetProperty('command', cmdline)
        self.porto_ct.Start()

    def check(self):
        if self.config.runtime_type == config_pb2.Config.PORTO:
            if not self.porto_ct:
                return False
            try:
                state = self.porto_ct.GetProperty('state')
                return state == 'running'
            except Exception:
                log.exception('check failed')
                return False

        elif self.config.runtime_type == config_pb2.Config.CHILD:
            try:
                child_state = self.child.poll()
                if child_state is None:
                    return True
                else:
                    return False
            except (Exception, AttributeError):
                log.exception('check failed')
                return False

    def spawn(self):
        if self.config.runtime_type == config_pb2.Config.PORTO:
            self.spawn_porto()
        elif self.config.runtime_type == config_pb2.Config.CHILD:
            self.spawn_child()

    def ensure(self):
        if not self.check():
            try:
                self.spawn()
                return True
            except Exception:
                log.exception('spawn failed')
                return False
        else:
            return True
