import argparse
import subprocess
import sys
import json


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument("--quota-mover", required=True, help="Path to quota mover binary.")
    parser.add_argument("--yp-util", required=True, help="Path to yp-util binary.")
    parser.add_argument("--from-service", required=True, help="From service.")
    parser.add_argument("--to-service", required=True, help="To service.")
    parser.add_argument("--resource", action="append", default=[], help="Move resource, options: cpu, memory, hdd, hdd_bw, ssd, ssd_bw, net_bw, all")
    parser.add_argument("--recursive", action="store_true", default=False, help="Recursive.")
    parser.add_argument("--cluster", required=True, help="YP cluster.")
    parser.add_argument("--yp-segment", default="default", help="YP sergment.")
    parser.add_argument("--execute", action="store_true", default=False, help="Really move the quota.")
    return parser.parse_args(argv)


QUOTA_CONVERTER = {
    "cpu": ("cpu_capacity:{}:millicores", lambda free: free["vcpu"]),
    "memory": ("memory_limit:{}:bytes", lambda free: free["memory"]),
    "hdd": ("hdd_disk_capacity:{}:bytes", lambda free: free["hdd"]),
    "hdd_bw": ("hdd_disk_bandwidth:{}:bytesPerSecond", lambda free: free["hdd_bw"]),
    "ssd": ("ssd_disk_capacity:{}:bytes", lambda free: free["ssd"]),
    "ssd_bw": ("ssd_disk_bandwidth:{}:bytesPerSecond", lambda free: free["ssd_bw"]),
    "net_bw": ("network_bandwidth:{}:bytesPerSecond", lambda free: free["net_bw"]),
}


def quota_mover_argv(free, resource_type, args):
    argv = [
        "--move",
        "--rise",
        "--provider", "yp-prod",
        "--from-service", free["account_id"].split(":")[-1],
        "--to-service", args.to_service
    ]

    format, count_fn = QUOTA_CONVERTER[resource_type]
    count = count_fn(free)
    if count <= 0:
        return None
    argv += ["--resource", format.format(count)]

    argv += ["--segments", args.cluster, args.yp_segment]

    return argv
    


def main(argv):
    args = parse_args(argv)
    
    yp_util_stdout = subprocess.check_output([
        args.yp_util,
        "account", "explain", "abc:service:{}".format(args.from_service),
        "--cluster", args.cluster,
        "--format", "json"] + (["--recursive"] if args.recursive else []))
    frees = json.loads(yp_util_stdout)["accounts_frees"]

    resources = set(args.resource)
    if "all" in resources:
        for convert in QUOTA_CONVERTER:
            resources.add(convert)
        resources.remove("all")

    for free in frees:
        if free["segment"] != args.yp_segment:
            continue

        account_id = free["account_id"].split(":")[-1]
        if account_id  == args.to_service:
            continue

        print("Preparing to move resources from {} ({}) to {}...".format(account_id, free["slug"], args.to_service))

        for resource in resources:
            argv = quota_mover_argv(free, resource, args)
            if argv:
                print("Moving {}, argv: '{}'".format(resource, " ".join(argv)))
                if args.execute:
                    subprocess.check_output([args.quota_mover] + argv)
            else:
                print("Nothing to move for {}".format(resource))



if __name__ == "__main__":
    main(sys.argv[1:])
