#include <infra/pod_agent/libs/daemon/tests/test_lib/test_canon.h>
#include <infra/pod_agent/libs/daemon/tests/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/dirut.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/join.h>

namespace NInfra::NPodAgent::NDaemonTest  {

class ITestInContainerCanon: public ITestCanon {
public:
    ITestInContainerCanon(
        const TString& testName
        , TDuration workDuration
    )
        : ITestCanon(testName)
        , WorkDuration_(workDuration)
        , NamespaceRoot_(TPortoContainerName::NoEscape(TEST_PREFIX + "_namespace_" + ToString(TThread::CurrentThreadId()) + "_" + TestName_))
        , OuterContainerName_({NamespaceRoot_, TPortoContainerName("pod_agent")})
        , LoggerFilePath_(NFs::CurrentWorkingDirectory() + "/" + testName + "_" + ToString(TThread::CurrentThreadId()) + "_eventlog")
    {
    }

    void Init() final {
        Params_.LoggerBackend = "FILE";
        Params_.LoggerLevel = "DEBUG";
        Params_.LoggerFilePath = LoggerFilePath_;
        // Decrease length of nested containers prefix because full name of container is over porto limit
        Params_.ContainersPrefix = "pod_agent_";

        TString commandString = JoinSeq(" ", Params_.GetPodAgentParamsVector());

        PortoClient_->Create(NamespaceRoot_).Success();
        PortoClient_->SetProperty(NamespaceRoot_, EPortoContainerProperty::EnablePorto, ToString(EEnablePorto::Isolate));

        PortoClient_->Create(OuterContainerName_).Success();
        PortoClient_->SetProperty(OuterContainerName_, EPortoContainerProperty::Command, commandString).Success();
        PortoClient_->SetProperty(OuterContainerName_, EPortoContainerProperty::Isolate, "false").Success();

        // Enable cpu and memory controllers
        PortoClient_->SetProperty(OuterContainerName_, EPortoContainerProperty::CpuLimit, "0c").Success();
        PortoClient_->SetProperty(OuterContainerName_, EPortoContainerProperty::MemoryLimit, "0").Success();

        PortoClient_->Start(OuterContainerName_).Success();

        WaitForReadiness();
    }

    void Finish() final {
        DestroyContainersWithPrefix(PortoClient_, TString(NamespaceRoot_));
        RemoveVolumes();
        RemoveLayers();
        RemoveStorages();
    }

    virtual API::TPodAgentRequest GetSpec() = 0;

    void DoTest() final {
        auto workloadSpec = GetSpec();
        Handle_.UpdatePodAgentRequest(workloadSpec);

        auto readyHook = [](const API::TPodAgentStatus& status) {
            return CheckAllObjectsReady(status) && status.workloads()[0].liveness_status().container_status().current().state() == API::EContainerState_WAITING_RESTART;
        };

        API::TPodAgentStatus workloadRet = UpdatePodAgentRequestAndWaitForReady(workloadSpec, 600, readyHook);

        PreparationTime_ = FromString(PortoClient_->GetProperty(OuterContainerName_, EPortoContainerProperty::Time).Success());
        size_t oldCpuUsageRaw = FromString(PortoClient_->GetProperty(OuterContainerName_, EPortoContainerProperty::CpuUsage).Success());
        PreparationCpuUsage_ = oldCpuUsageRaw / PreparationTime_ / 1e9;
        TMap<TString, size_t> oldPortoCallMap = GetPortoCallsMap();
        size_t oldPortoCalls = 0;
        for (const auto& [name, value]: oldPortoCallMap) {
            if (name == PORTO_TOTAL_REQUESTS_SENSOR_NAME) {
                oldPortoCalls = value;
            }
            PreparationPortoRpsMap_[name] = value / PreparationTime_;
        }
        UNIT_ASSERT_C(oldPortoCalls > 0, "Something went wrong, totalPortoCalls for preparation can't be zero");

        PreparationPortoRPS_ = oldPortoCalls / PreparationTime_;

        ui64 oldLogFileSize = TFileStat(LoggerFilePath_).Size;
        PreparationLogFileSizeKBPerSecond_ = oldLogFileSize / PreparationTime_ / (1 << 10);

        Sleep(WorkDuration_);

        MaxRss_ = FromString(PortoClient_->GetProperty(OuterContainerName_, EPortoContainerProperty::MaxRss).Success());
        double totalCpuUsageRaw = FromString(PortoClient_->GetProperty(OuterContainerName_, EPortoContainerProperty::CpuUsage).Success());
        TMap<TString, size_t> newPortoCallMap = GetPortoCallsMap();
        size_t totalPortoCalls = 0;
        for (const auto& [name, value]: newPortoCallMap) {
            if (name == PORTO_TOTAL_REQUESTS_SENSOR_NAME) {
                totalPortoCalls = value;
            }
            PortoRpsMap_[name] = 1.0 * (value - oldPortoCallMap[name]) / WorkDuration_.Seconds();
        }
        UNIT_ASSERT_C(totalPortoCalls > 0, "Something went wrong, totalPortoCalls can't be zero");

        PortoRPS_ = 1.0 * (totalPortoCalls - oldPortoCalls) / WorkDuration_.Seconds();
        CpuUsage_ = 1.0 * (totalCpuUsageRaw - oldCpuUsageRaw) / WorkDuration_.Seconds() / 1e9;

        ui64 totalLogFileSize = TFileStat(LoggerFilePath_).Size;
        LogFileSizeKBPerSecond_ = (double)(totalLogFileSize - oldLogFileSize) / WorkDuration_.Seconds() / (1 << 10);

        Handle_.Shutdown();
        for (i32 i = 0; i < 10; ++i) {
            Sleep(TDuration::MilliSeconds(500));
            ContainerState_ = PortoClient_->GetProperty(OuterContainerName_, EPortoContainerProperty::State).Success();
            if (ContainerState_ == "dead") {
                break;
            }
        }
    }

public:
    double MaxRss_;
    double PreparationCpuUsage_;
    double PreparationPortoRPS_;
    double CpuUsage_;
    double PortoRPS_;
    double PreparationTime_;
    TString ContainerState_;
    TMap<TString, double> PortoRpsMap_;
    TMap<TString, double> PreparationPortoRpsMap_;
    double PreparationLogFileSizeKBPerSecond_;
    double LogFileSizeKBPerSecond_;

private:
    static inline const TString PORTO_TOTAL_REQUESTS_SENSOR_NAME = "requests_total";

    const TDuration WorkDuration_;

    const TPortoContainerName NamespaceRoot_;
    const TPortoContainerName OuterContainerName_;
    const TString LoggerFilePath_;
};

Y_UNIT_TEST_SUITE(PodAgentDaemonTest) {

Y_UNIT_TEST(TestInContainer) {
    class TTest : public ITestInContainerCanon {
    public:
        TTest(): ITestInContainerCanon(
            "TestInContainer"
            , TDuration::Seconds(30)
        ) {
        }

        API::TPodAgentRequest GetSpec() override {
            return SpecHolder_.GetTestInContainerWorkloadSpec();
        }
    };

    TTest test;
    test.Test();

    UNIT_ADD_METRIC("max_rss", test.MaxRss_);
    UNIT_ADD_METRIC("cpu_usage", test.CpuUsage_);
    UNIT_ADD_METRIC("porto_rps", test.PortoRPS_);
    UNIT_ADD_METRIC("preparation_cpu_usage", test.PreparationCpuUsage_);
    UNIT_ADD_METRIC("preparation_porto_rps", test.PreparationPortoRPS_);
    UNIT_ADD_METRIC("preparation_time", test.PreparationTime_);
    for (auto& sensor: test.PortoRpsMap_) {
        UNIT_ADD_METRIC("porto_rps_" + sensor.first, sensor.second);
    }
    for (auto& sensor: test.PreparationPortoRpsMap_) {
        UNIT_ADD_METRIC("preparation_porto_rps_" + sensor.first, sensor.second);
    }
    UNIT_ADD_METRIC("log_file_size_kb_per_second", test.LogFileSizeKBPerSecond_);
    UNIT_ADD_METRIC("preparation_log_file_size_kb_per_second", test.PreparationLogFileSizeKBPerSecond_);

    UNIT_ASSERT_C(test.PreparationCpuUsage_ < 0.165, test.PreparationCpuUsage_);
    UNIT_ASSERT_C(test.PreparationPortoRPS_ > 0, test.PreparationPortoRPS_);
    UNIT_ASSERT_C(test.PreparationPortoRPS_ < 160, test.PreparationPortoRPS_);

    UNIT_ASSERT_C(test.PortoRPS_ < 10, test.PortoRPS_);
    UNIT_ASSERT_C(test.MaxRss_ < 50 * ((size_t)1 << 20), test.MaxRss_);
    UNIT_ASSERT_C(test.CpuUsage_ < 0.02, test.CpuUsage_);

    UNIT_ASSERT_EQUAL_C(test.ContainerState_, "dead", test.ContainerState_);
}

Y_UNIT_TEST(TestInContainerBigSpec) {
    class TTest : public ITestInContainerCanon {
    public:
        TTest(): ITestInContainerCanon(
            "TestInContainerBigSpec"
            , TDuration::Seconds(30)
        ) {
        }

        API::TPodAgentRequest GetSpec() override {
            return SpecHolder_.GetTestInContainerBigWorkloadSpec();
        }
    };

    TTest test;
    test.Test();

    UNIT_ADD_METRIC("max_rss", test.MaxRss_);
    UNIT_ADD_METRIC("cpu_usage", test.CpuUsage_);
    UNIT_ADD_METRIC("porto_rps", test.PortoRPS_);
    UNIT_ADD_METRIC("preparation_cpu_usage", test.PreparationCpuUsage_);
    UNIT_ADD_METRIC("preparation_porto_rps", test.PreparationPortoRPS_);
    UNIT_ADD_METRIC("preparation_time", test.PreparationTime_);
    for (auto& sensor: test.PortoRpsMap_) {
        UNIT_ADD_METRIC("porto_rps_" + sensor.first, sensor.second);
    }
    for (auto& sensor: test.PreparationPortoRpsMap_) {
        UNIT_ADD_METRIC("preparation_porto_rps_" + sensor.first, sensor.second);
    }
    UNIT_ADD_METRIC("log_file_size_kb_per_second", test.LogFileSizeKBPerSecond_);
    UNIT_ADD_METRIC("preparation_log_file_size_kb_per_second", test.PreparationLogFileSizeKBPerSecond_);

    UNIT_ASSERT_C(test.PreparationCpuUsage_ < 0.4, test.PreparationCpuUsage_);
    UNIT_ASSERT_C(test.PreparationPortoRPS_ > 0, test.PreparationPortoRPS_);
    UNIT_ASSERT_C(test.PreparationPortoRPS_ < 650, test.PreparationPortoRPS_);

    UNIT_ASSERT_C(test.PortoRPS_ < 150, test.PortoRPS_);
    UNIT_ASSERT_C(test.MaxRss_ < 100 * ((size_t)1 << 20), test.MaxRss_);
    UNIT_ASSERT_C(test.CpuUsage_ < 0.06, test.CpuUsage_);

    UNIT_ASSERT_EQUAL_C(test.ContainerState_, "dead", test.ContainerState_);
}

Y_UNIT_TEST(TestDiskUsage) {
    TFileStat stat(BinaryPath("infra/pod_agent/daemons/pod_agent/pod_agent"));

    UNIT_ADD_METRIC("disk_usage", stat.Size);
    UNIT_ASSERT_C(stat.Size < 502 * ((size_t)1 << 20), stat.Size);  // 502MB
}

}

} // namespace NInfra::NPodAgent::NDaemonTest
