#pragma once

#include "common.h"
#include "compare_times_types.h"
#include "porto_basic_node.h"

#include <infra/pod_agent/libs/posix_worker/posix_worker.h>

namespace NInfra::NPodAgent {

template<class PosixTimeResult>
class TModifiedBeforePortoContainerStartBasicNode : public TPortoBasicNode<TExpected<TCompareTimesResult, TModifiedTimeError>> {
public:
    TModifiedBeforePortoContainerStartBasicNode(
        const TBasicTreeNodeDescriptor& descriptor
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , const TPortoContainerName& containerName
        , const TString& path
    )
        : TPortoBasicNode(descriptor, porto)
        , Posix_(posixWorker)
        , ContainerName_(containerName)
        , Path_(path)
    {
        Y_ENSURE(Posix_, "Posix not defined for TModifiedBeforePortoContainerStartBasicNode");
    }

protected:
    virtual NThreading::TFuture<PosixTimeResult> GetPosixTimeResult() = 0;
    virtual TExpected<time_t, TPosixError> GetModificationTimeFromPosixResult(const PosixTimeResult& posixTimeResult) = 0;

private:
    virtual TExpected<TFuture<TExpected<TCompareTimesResult, TModifiedTimeError>>, TTickResult> PortoCall(TTickContextPtr /*context*/) override final {
        return GetPosixTimeResult().Apply([this](const auto& posixModificationTimeResult){
            return Porto_->GetProperty(ContainerName_, EPortoContainerProperty::StartTimeRaw).Apply([this, posixModificationTimeResult](const auto& startTimeResult){
                return CompareTimes(startTimeResult.GetValue(), posixModificationTimeResult.GetValue());
            });
        });
    }

    TExpected<TCompareTimesResult, TModifiedTimeError> CompareTimes(
        const TExpected<TString, TPortoError>& containerStartTimeResult
        , const PosixTimeResult& posixModificationTimeResult
     ) {
        if (!(bool)(containerStartTimeResult))
            return TModifiedTimeError(containerStartTimeResult.Error());

        auto modificationTimeResult = GetModificationTimeFromPosixResult(posixModificationTimeResult);
        if (!(bool)(modificationTimeResult))
            return TModifiedTimeError(modificationTimeResult.Error());

        try {
            time_t containerStartTime = FromString<time_t>(containerStartTimeResult.Success());
            time_t modificationTime = modificationTimeResult.Success();
            if (modificationTime <= containerStartTime)
                return TCompareTimesResult { true };
            else
                return TCompareTimesResult { false, TStringBuilder() << "container start time: " << containerStartTime << ", modification time: " << modificationTime };

        } catch (TFromStringException& e) {
            return TModifiedTimeError(TPortoError{EPortoError::InvalidValue, "", "", TStringBuilder() << "Could not parse DateTime from:" << "'" << containerStartTimeResult.Success() << "'" });
        }
    }

    virtual TTickResult ProcessPortoResult(TTickContextPtr /*context*/, TExpected<TCompareTimesResult, TModifiedTimeError>& result) override final {
        if (!result) {
            auto error = result.Error();
            if (std::holds_alternative<TPosixError>(error))
                return TNodeError{ToString(std::get<TPosixError>(error))};

            if (std::holds_alternative<TPortoError>(error))
                return TNodeError{ToString(std::get<TPortoError>(error))};

            return TNodeError{"unknown error"};
        }

        return result.Success().Status
               ? TNodeSuccess(ENodeStatus::SUCCESS)
               : TNodeSuccess(ENodeStatus::FAILURE, TStringBuilder() << "modified after container start: " << Path_ << ", " << result.Success().Message);
    }

protected:
    TPosixWorkerPtr Posix_;
    const TPortoContainerName ContainerName_;
    const TString Path_;
};

} // namespace NInfra::NPodAgent
