#include "mem_selector_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestMemSelectorNode {

Y_UNIT_TEST_SUITE(MemSelectorNodeSuite) {

static TLogger logger({});

TMemSelectorNodePtr SingleChildSelector(TTickResult tickResult) {
    return new TMemSelectorNode({1, "title"}, {new TMockNode({1, "title"}, tickResult)});
}

Y_UNIT_TEST(TestWithOneFailedChild) {
    TTickResult expected = TNodeError{"test error message"};

    auto selector = SingleChildSelector(expected);

    auto result = selector->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(expected, result);
}

Y_UNIT_TEST(TestWithOneSuccessChild) {
    ENodeStatus const goodStatuses[] = {ENodeStatus::SUCCESS, ENodeStatus::FAILURE, ENodeStatus::RUNNING};

    for (auto&& status : goodStatuses) {
        TTickResult expected(status);
        auto selector = SingleChildSelector(expected);

        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(expected, result);
    }
}

Y_UNIT_TEST(TesWithOneUnexpextedChildStatus) {
    TTickResult idleChild(ENodeStatus::IDLE);

    auto selector = SingleChildSelector(idleChild);

    auto result = selector->Tick(MockTickContext(logger));
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "unexpected status");
}

Y_UNIT_TEST(TestTwoChildrenSuccessAndSuccess) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }
}

Y_UNIT_TEST(TestTwoChildrenFailureAndSuccess) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }
}

Y_UNIT_TEST(TestTwoChildrenSuccessAndFailure) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }
}

Y_UNIT_TEST(TestTwoChildrenFailureAndFailure) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }
}

Y_UNIT_TEST(TestTwoChildrenSuccessAndRunning) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::RUNNING))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }
}

Y_UNIT_TEST(TestTwoChildrenRunningAndSuccess) {
    TMemSelectorNodePtr selector = new TMemSelectorNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::RUNNING))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
        }
    );

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }

    {
        auto result = selector->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }
}

}

} // namespace NInfra::NPodAgent::NTestMemSelectorNode


